# Copyright 2020 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require 'google/apis/core/base_service'
require 'google/apis/core/json_representation'
require 'google/apis/core/hashable'
require 'google/apis/errors'

module Google
  module Apis
    module IamV1
      # Identity and Access Management (IAM) API
      #
      # Manages identity and access control for Google Cloud resources, including the
      #  creation of service accounts, which you can use to authenticate to Google and
      #  make API calls. Enabling this API also enables the IAM Service Account
      #  Credentials API (iamcredentials.googleapis.com). However, disabling this API
      #  doesn't disable the IAM Service Account Credentials API.
      #
      # @example
      #    require 'google/apis/iam_v1'
      #
      #    Iam = Google::Apis::IamV1 # Alias the module
      #    service = Iam::IamService.new
      #
      # @see https://cloud.google.com/iam/
      class IamService < Google::Apis::Core::BaseService
        DEFAULT_ENDPOINT_TEMPLATE = "https://iam.$UNIVERSE_DOMAIN$/"

        # @return [String]
        #  API key. Your API key identifies your project and provides you with API access,
        #  quota, and reports. Required unless you provide an OAuth 2.0 token.
        attr_accessor :key

        # @return [String]
        #  Available to use for quota purposes for server-side applications. Can be any
        #  arbitrary string assigned to a user, but should not exceed 40 characters.
        attr_accessor :quota_user

        def initialize
          super(DEFAULT_ENDPOINT_TEMPLATE, '',
                client_name: 'google-apis-iam_v1',
                client_version: Google::Apis::IamV1::GEM_VERSION)
          @batch_path = 'batch'
        end
        
        # Lints, or validates, an IAM policy. Currently checks the google.iam.v1.Binding.
        # condition field, which contains a condition expression for a role binding.
        # Successful calls to this method always return an HTTP `200 OK` status code,
        # even if the linter detects an issue in the IAM policy.
        # @param [Google::Apis::IamV1::LintPolicyRequest] lint_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::LintPolicyResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::LintPolicyResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def lint_iam_policy_policy(lint_policy_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/iamPolicies:lintPolicy', options)
          command.request_representation = Google::Apis::IamV1::LintPolicyRequest::Representation
          command.request_object = lint_policy_request_object
          command.response_representation = Google::Apis::IamV1::LintPolicyResponse::Representation
          command.response_class = Google::Apis::IamV1::LintPolicyResponse
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns a list of services that allow you to opt into audit logs that are not
        # generated by default. To learn more about audit logs, see the [Logging
        # documentation](https://cloud.google.com/logging/docs/audit).
        # @param [Google::Apis::IamV1::QueryAuditableServicesRequest] query_auditable_services_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::QueryAuditableServicesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::QueryAuditableServicesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def query_iam_policy_auditable_services(query_auditable_services_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/iamPolicies:queryAuditableServices', options)
          command.request_representation = Google::Apis::IamV1::QueryAuditableServicesRequest::Representation
          command.request_object = query_auditable_services_request_object
          command.response_representation = Google::Apis::IamV1::QueryAuditableServicesResponse::Representation
          command.response_class = Google::Apis::IamV1::QueryAuditableServicesResponse
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new WorkforcePool. You cannot reuse the name of a deleted pool until
        # 30 days after deletion.
        # @param [String] location
        #   Optional. The location of the pool to create. Format: `locations/`location``.
        # @param [Google::Apis::IamV1::WorkforcePool] workforce_pool_object
        # @param [String] workforce_pool_id
        #   Optional. The ID to use for the pool, which becomes the final component of the
        #   resource name. The IDs must be a globally unique string of 6 to 63 lowercase
        #   letters, digits, or hyphens. It must start with a letter, and cannot have a
        #   trailing hyphen. The prefix `gcp-` is reserved for use by Google, and may not
        #   be specified.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_location_workforce_pool(location, workforce_pool_object = nil, workforce_pool_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+location}/workforcePools', options)
          command.request_representation = Google::Apis::IamV1::WorkforcePool::Representation
          command.request_object = workforce_pool_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['location'] = location unless location.nil?
          command.query['workforcePoolId'] = workforce_pool_id unless workforce_pool_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a WorkforcePool. You cannot use a deleted WorkforcePool to exchange
        # external credentials for Google Cloud credentials. However, deletion does not
        # revoke credentials that have already been issued. Credentials issued for a
        # deleted pool do not grant access to resources. If the pool is undeleted, and
        # the credentials are not expired, they grant access again. You can undelete a
        # pool for 30 days. After 30 days, deletion is permanent. You cannot update
        # deleted pools. However, you can view and list them.
        # @param [String] name
        #   Required. The name of the pool to delete. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id``
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_location_workforce_pool(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an individual WorkforcePool.
        # @param [String] name
        #   Required. The name of the pool to retrieve. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id``
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePool] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePool]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_location_workforce_pool(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkforcePool::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePool
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets IAM policies on a WorkforcePool.
        # @param [String] resource
        #   REQUIRED: The resource for which the policy is being requested. See [Resource
        #   names](https://cloud.google.com/apis/design/resource_names) for the
        #   appropriate value for this field.
        # @param [Google::Apis::IamV1::GetIamPolicyRequest] get_iam_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_workforce_pool_iam_policy(resource, get_iam_policy_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:getIamPolicy', options)
          command.request_representation = Google::Apis::IamV1::GetIamPolicyRequest::Representation
          command.request_object = get_iam_policy_request_object
          command.response_representation = Google::Apis::IamV1::Policy::Representation
          command.response_class = Google::Apis::IamV1::Policy
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all non-deleted WorkforcePools under the specified parent. If `
        # show_deleted` is set to `true`, then deleted pools are also listed.
        # @param [String] location
        #   The location of the pool. Format: `locations/`location``.
        # @param [Fixnum] page_size
        #   The maximum number of pools to return. The default value is 50. The maximum
        #   value is 100.
        # @param [String] page_token
        #   A page token, received from a previous `ListWorkforcePools` call. Provide this
        #   to retrieve the subsequent page.
        # @param [String] parent
        #   Required. The parent resource to list pools for. Format: `organizations/`org-
        #   id``.
        # @param [Boolean] show_deleted
        #   Whether to return soft-deleted pools.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkforcePoolsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkforcePoolsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_location_workforce_pools(location, page_size: nil, page_token: nil, parent: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+location}/workforcePools', options)
          command.response_representation = Google::Apis::IamV1::ListWorkforcePoolsResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkforcePoolsResponse
          command.params['location'] = location unless location.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['parent'] = parent unless parent.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an existing WorkforcePool.
        # @param [String] name
        #   Identifier. The resource name of the pool. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id``
        # @param [Google::Apis::IamV1::WorkforcePool] workforce_pool_object
        # @param [String] update_mask
        #   Required. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_location_workforce_pool(name, workforce_pool_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::WorkforcePool::Representation
          command.request_object = workforce_pool_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets IAM policies on a WorkforcePool.
        # @param [String] resource
        #   REQUIRED: The resource for which the policy is being specified. See [Resource
        #   names](https://cloud.google.com/apis/design/resource_names) for the
        #   appropriate value for this field.
        # @param [Google::Apis::IamV1::SetIamPolicyRequest] set_iam_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_workforce_pool_iam_policy(resource, set_iam_policy_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:setIamPolicy', options)
          command.request_representation = Google::Apis::IamV1::SetIamPolicyRequest::Representation
          command.request_object = set_iam_policy_request_object
          command.response_representation = Google::Apis::IamV1::Policy::Representation
          command.response_class = Google::Apis::IamV1::Policy
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the caller's permissions on the WorkforcePool. If the pool doesn't
        # exist, this call returns an empty set of permissions. It doesn't return a `
        # NOT_FOUND` error.
        # @param [String] resource
        #   REQUIRED: The resource for which the policy detail is being requested. See [
        #   Resource names](https://cloud.google.com/apis/design/resource_names) for the
        #   appropriate value for this field.
        # @param [Google::Apis::IamV1::TestIamPermissionsRequest] test_iam_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::TestIamPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::TestIamPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_workforce_pool_iam_permissions(resource, test_iam_permissions_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:testIamPermissions', options)
          command.request_representation = Google::Apis::IamV1::TestIamPermissionsRequest::Representation
          command.request_object = test_iam_permissions_request_object
          command.response_representation = Google::Apis::IamV1::TestIamPermissionsResponse::Representation
          command.response_class = Google::Apis::IamV1::TestIamPermissionsResponse
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a WorkforcePool, as long as it was deleted fewer than 30 days ago.
        # @param [String] name
        #   Required. The name of the pool to undelete. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id``
        # @param [Google::Apis::IamV1::UndeleteWorkforcePoolRequest] undelete_workforce_pool_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workforce_pool(name, undelete_workforce_pool_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkforcePoolRequest::Representation
          command.request_object = undelete_workforce_pool_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_location_workforce_pool_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new WorkforcePoolProvider in a WorkforcePool. You cannot reuse the
        # name of a deleted provider until 30 days after deletion.
        # @param [String] parent
        #   Required. The pool to create this provider in. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id``
        # @param [Google::Apis::IamV1::WorkforcePoolProvider] workforce_pool_provider_object
        # @param [String] workforce_pool_provider_id
        #   Required. The ID for the provider, which becomes the final component of the
        #   resource name. This value must be 4-32 characters, and may contain the
        #   characters [a-z0-9-]. The prefix `gcp-` is reserved for use by Google, and may
        #   not be specified.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_location_workforce_pool_provider(parent, workforce_pool_provider_object = nil, workforce_pool_provider_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/providers', options)
          command.request_representation = Google::Apis::IamV1::WorkforcePoolProvider::Representation
          command.request_object = workforce_pool_provider_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['parent'] = parent unless parent.nil?
          command.query['workforcePoolProviderId'] = workforce_pool_provider_id unless workforce_pool_provider_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a WorkforcePoolProvider. Deleting a provider does not revoke
        # credentials that have already been issued; they continue to grant access. You
        # can undelete a provider for 30 days. After 30 days, deletion is permanent. You
        # cannot update deleted providers. However, you can view and list them.
        # @param [String] name
        #   Required. The name of the provider to delete. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id`/providers/`provider_id``
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_location_workforce_pool_provider(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an individual WorkforcePoolProvider.
        # @param [String] name
        #   Required. The name of the provider to retrieve. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id`/providers/`provider_id``
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProvider] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProvider]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_location_workforce_pool_provider(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProvider::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProvider
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all non-deleted WorkforcePoolProviders in a WorkforcePool. If `
        # show_deleted` is set to `true`, then deleted providers are also listed.
        # @param [String] parent
        #   Required. The pool to list providers for. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id``
        # @param [Fixnum] page_size
        #   The maximum number of providers to return. If unspecified, at most 50
        #   providers are returned. The maximum value is 100; values above 100 are
        #   truncated to 100.
        # @param [String] page_token
        #   A page token, received from a previous `ListWorkforcePoolProviders` call.
        #   Provide this to retrieve the subsequent page.
        # @param [Boolean] show_deleted
        #   Whether to return soft-deleted providers.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkforcePoolProvidersResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkforcePoolProvidersResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_location_workforce_pool_providers(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/providers', options)
          command.response_representation = Google::Apis::IamV1::ListWorkforcePoolProvidersResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkforcePoolProvidersResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an existing WorkforcePoolProvider.
        # @param [String] name
        #   Identifier. The resource name of the provider. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id`/providers/`provider_id``
        # @param [Google::Apis::IamV1::WorkforcePoolProvider] workforce_pool_provider_object
        # @param [String] update_mask
        #   Required. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_location_workforce_pool_provider(name, workforce_pool_provider_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::WorkforcePoolProvider::Representation
          command.request_object = workforce_pool_provider_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a WorkforcePoolProvider, as long as it was deleted fewer than 30
        # days ago.
        # @param [String] name
        #   Required. The name of the provider to undelete. Format: `locations/`location`/
        #   workforcePools/`workforce_pool_id`/providers/`provider_id``
        # @param [Google::Apis::IamV1::UndeleteWorkforcePoolProviderRequest] undelete_workforce_pool_provider_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workforce_pool_provider(name, undelete_workforce_pool_provider_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkforcePoolProviderRequest::Representation
          command.request_object = undelete_workforce_pool_provider_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new WorkforcePoolProviderKey in a WorkforcePoolProvider.
        # @param [String] parent
        #   Required. The provider to create this key in.
        # @param [Google::Apis::IamV1::WorkforcePoolProviderKey] workforce_pool_provider_key_object
        # @param [String] workforce_pool_provider_key_id
        #   Required. The ID to use for the key, which becomes the final component of the
        #   resource name. This value must be 4-32 characters, and may contain the
        #   characters [a-z0-9-].
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_location_workforce_pool_provider_key(parent, workforce_pool_provider_key_object = nil, workforce_pool_provider_key_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/keys', options)
          command.request_representation = Google::Apis::IamV1::WorkforcePoolProviderKey::Representation
          command.request_object = workforce_pool_provider_key_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['parent'] = parent unless parent.nil?
          command.query['workforcePoolProviderKeyId'] = workforce_pool_provider_key_id unless workforce_pool_provider_key_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a WorkforcePoolProviderKey. You can undelete a key for 30 days. After
        # 30 days, deletion is permanent.
        # @param [String] name
        #   Required. The name of the key to delete.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_location_workforce_pool_provider_key(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets a WorkforcePoolProviderKey.
        # @param [String] name
        #   Required. The name of the key to retrieve.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderKey] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderKey]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_location_workforce_pool_provider_key(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderKey::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderKey
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all non-deleted WorkforcePoolProviderKeys in a WorkforcePoolProvider. If
        # `show_deleted` is set to `true`, then deleted keys are also listed.
        # @param [String] parent
        #   Required. The provider resource to list encryption keys for. Format: `
        #   locations/`location`/workforcePools/`workforce_pool_id`/providers/`provider_id`
        #   `
        # @param [Fixnum] page_size
        #   The maximum number of keys to return. If unspecified, all keys are returned.
        #   The maximum value is 10; values above 10 are truncated to 10.
        # @param [String] page_token
        #   A page token, received from a previous `ListWorkforcePoolProviderKeys` call.
        #   Provide this to retrieve the subsequent page.
        # @param [Boolean] show_deleted
        #   Whether to return soft-deleted keys.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkforcePoolProviderKeysResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkforcePoolProviderKeysResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_location_workforce_pool_provider_keys(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/keys', options)
          command.response_representation = Google::Apis::IamV1::ListWorkforcePoolProviderKeysResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkforcePoolProviderKeysResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a WorkforcePoolProviderKey, as long as it was deleted fewer than 30
        # days ago.
        # @param [String] name
        #   Required. The name of the key to undelete.
        # @param [Google::Apis::IamV1::UndeleteWorkforcePoolProviderKeyRequest] undelete_workforce_pool_provider_key_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workforce_pool_provider_key(name, undelete_workforce_pool_provider_key_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkforcePoolProviderKeyRequest::Representation
          command.request_object = undelete_workforce_pool_provider_key_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_location_workforce_pool_provider_key_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_location_workforce_pool_provider_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Creates a new WorkforcePoolProviderScimTenant in a
        # WorkforcePoolProvider. You cannot reuse the name of a deleted SCIM tenant
        # until 30 days after deletion.
        # @param [String] parent
        #   Required. Gemini Enterprise only. The parent to create SCIM tenant. Format: '
        #   locations/`location`/workforcePools/`workforce_pool`/providers/`provider`'
        # @param [Google::Apis::IamV1::WorkforcePoolProviderScimTenant] workforce_pool_provider_scim_tenant_object
        # @param [String] workforce_pool_provider_scim_tenant_id
        #   Required. Gemini Enterprise only. The ID to use for the SCIM tenant, which
        #   becomes the final component of the resource name. This value should be 4-32
        #   characters, containing the characters [a-z0-9-].
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimTenant] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimTenant]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_location_workforce_pool_provider_scim_tenant(parent, workforce_pool_provider_scim_tenant_object = nil, workforce_pool_provider_scim_tenant_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/scimTenants', options)
          command.request_representation = Google::Apis::IamV1::WorkforcePoolProviderScimTenant::Representation
          command.request_object = workforce_pool_provider_scim_tenant_object
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimTenant::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimTenant
          command.params['parent'] = parent unless parent.nil?
          command.query['workforcePoolProviderScimTenantId'] = workforce_pool_provider_scim_tenant_id unless workforce_pool_provider_scim_tenant_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Deletes a WorkforcePoolProviderScimTenant. You can
        # undelete a SCIM tenant for 30 days. After 30 days, deletion is permanent. You
        # cannot update deleted SCIM tenants. However, you can view and list them.
        # @param [String] name
        #   Required. Gemini Enterprise only. The name of the SCIM tenant to delete.
        #   Format: `locations/`location`/workforcePools/`workforce_pool`/providers/`
        #   provider`/scimTenants/`scim_tenant``
        # @param [Boolean] hard_delete
        #   Optional. Deletes the SCIM tenant immediately. This operation cannot be undone.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimTenant] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimTenant]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_location_workforce_pool_provider_scim_tenant(name, hard_delete: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimTenant::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimTenant
          command.params['name'] = name unless name.nil?
          command.query['hardDelete'] = hard_delete unless hard_delete.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Gets an individual WorkforcePoolProviderScimTenant.
        # @param [String] name
        #   Required. Gemini Enterprise only. The name of the SCIM tenant to retrieve.
        #   Format: `locations/`location`/workforcePools/`workforce_pool`/providers/`
        #   provider`/scimTenants/`scim_tenant``
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimTenant] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimTenant]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_location_workforce_pool_provider_scim_tenant(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimTenant::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimTenant
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Lists all non-deleted WorkforcePoolProviderScimTenants
        # in a WorkforcePoolProvider. If `show_deleted` is set to `true`, then deleted
        # SCIM tenants are also listed.
        # @param [String] parent
        #   Required. Gemini Enterprise only. The parent to list SCIM tenants. Format: '
        #   locations/`location`/workforcePools/`workforce_pool`/providers/`provider`'
        # @param [Fixnum] page_size
        #   Optional. Gemini Enterprise only. The maximum number of SCIM tenants to return.
        #   If unspecified, at most 50 SCIM tenants will be returned. The maximum value
        #   is 100; values above 100 are truncated to 100.
        # @param [String] page_token
        #   Optional. Gemini Enterprise only. A page token, received from a previous `
        #   ListScimTenants` call. Provide this to retrieve the subsequent page.
        # @param [Boolean] show_deleted
        #   Optional. Gemini Enterprise only. Whether to return soft-deleted SCIM tenants.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkforcePoolProviderScimTenantsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkforcePoolProviderScimTenantsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_location_workforce_pool_provider_scim_tenants(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/scimTenants', options)
          command.response_representation = Google::Apis::IamV1::ListWorkforcePoolProviderScimTenantsResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkforcePoolProviderScimTenantsResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Updates an existing WorkforcePoolProviderScimTenant.
        # @param [String] name
        #   Identifier. Gemini Enterprise only. The resource name of the SCIM Tenant.
        #   Format: `locations/`location`/workforcePools/`workforce_pool`/providers/ `
        #   workforce_pool_provider`/scimTenants/`scim_tenant``
        # @param [Google::Apis::IamV1::WorkforcePoolProviderScimTenant] workforce_pool_provider_scim_tenant_object
        # @param [String] update_mask
        #   Optional. Gemini Enterprise only. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimTenant] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimTenant]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_location_workforce_pool_provider_scim_tenant(name, workforce_pool_provider_scim_tenant_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::WorkforcePoolProviderScimTenant::Representation
          command.request_object = workforce_pool_provider_scim_tenant_object
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimTenant::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimTenant
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Undeletes a WorkforcePoolProviderScimTenant, that was
        # deleted fewer than 30 days ago.
        # @param [String] name
        #   Required. Gemini Enterprise only. The name of the SCIM tenant to undelete.
        #   Format: `locations/`location`/workforcePools/`workforce_pool`/providers/`
        #   provider`/scimTenants/`scim_tenant``
        # @param [Google::Apis::IamV1::UndeleteWorkforcePoolProviderScimTenantRequest] undelete_workforce_pool_provider_scim_tenant_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimTenant] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimTenant]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workforce_pool_provider_scim_tenant(name, undelete_workforce_pool_provider_scim_tenant_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkforcePoolProviderScimTenantRequest::Representation
          command.request_object = undelete_workforce_pool_provider_scim_tenant_request_object
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimTenant::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimTenant
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Creates a new WorkforcePoolProviderScimToken in a
        # WorkforcePoolProviderScimTenant. You cannot reuse the name of a deleted SCIM
        # token until 30 days after deletion.
        # @param [String] parent
        #   Required. Gemini Enterprise only. The parent tenant to create SCIM token.
        #   Format: 'locations/`location`/workforcePools/`workforce_pool`/providers/`
        #   provider`/scimTenants/`scim_tenant`'
        # @param [Google::Apis::IamV1::WorkforcePoolProviderScimToken] workforce_pool_provider_scim_token_object
        # @param [String] workforce_pool_provider_scim_token_id
        #   Required. Gemini Enterprise only. The ID to use for the SCIM token, which
        #   becomes the final component of the resource name. This value should be 4-32
        #   characters and follow the pattern: "([a-z]([a-z0-9\\-]`2,30`[a-z0-9]))"
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimToken] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimToken]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_location_workforce_pool_provider_scim_tenant_token(parent, workforce_pool_provider_scim_token_object = nil, workforce_pool_provider_scim_token_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/tokens', options)
          command.request_representation = Google::Apis::IamV1::WorkforcePoolProviderScimToken::Representation
          command.request_object = workforce_pool_provider_scim_token_object
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimToken::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimToken
          command.params['parent'] = parent unless parent.nil?
          command.query['workforcePoolProviderScimTokenId'] = workforce_pool_provider_scim_token_id unless workforce_pool_provider_scim_token_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Deletes a WorkforcePoolProviderScimToken. You can
        # undelete a SCIM token for 30 days. After 30 days, the SCIM token is
        # permanently deleted. You cannot update deleted SCIM tokens, however, you can
        # view and list them.
        # @param [String] name
        #   Required. Gemini Enterprise only. The name of the SCIM token to delete. Format:
        #   `locations/`location`/workforcePools/`workforce_pool`/providers/`provider`/
        #   scimTenants/`scim_tenant`/tokens/`token``
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimToken] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimToken]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_location_workforce_pool_provider_scim_tenant_token(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimToken::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimToken
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Gets an individual WorkforcePoolProviderScimToken.
        # @param [String] name
        #   Required. Gemini Enterprise only. The name of the SCIM token to retrieve.
        #   Format: `locations/`location`/workforcePools/`workforce_pool`/providers/`
        #   provider`/scimTenants/`scim_tenant`/tokens/`token``
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimToken] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimToken]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_location_workforce_pool_provider_scim_tenant_token(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimToken::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimToken
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Lists all non-deleted WorkforcePoolProviderScimTokenss
        # in a WorkforcePoolProviderScimTenant. If `show_deleted` is set to `true`, then
        # deleted SCIM tokens are also listed.
        # @param [String] parent
        #   Required. Gemini Enterprise only. The parent to list SCIM tokens. Format: '
        #   locations/`location`/workforcePools/`workforce_pool`/providers/`provider`/
        #   scimTenants/`scim_tenant`'
        # @param [Fixnum] page_size
        #   Optional. Gemini Enterprise only. The maximum number of SCIM tokens to return.
        #   If unspecified, at most 2 SCIM tokens will be returned.
        # @param [String] page_token
        #   Optional. Gemini Enterprise only. A page token, received from a previous `
        #   ListWorkforcePoolProviderScimTokens` call. Provide this to retrieve the
        #   subsequent page.
        # @param [Boolean] show_deleted
        #   Optional. Gemini Enterprise only. Whether to return soft-deleted SCIM tokens.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkforcePoolProviderScimTokensResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkforcePoolProviderScimTokensResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_location_workforce_pool_provider_scim_tenant_tokens(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/tokens', options)
          command.response_representation = Google::Apis::IamV1::ListWorkforcePoolProviderScimTokensResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkforcePoolProviderScimTokensResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Updates an existing WorkforcePoolProviderScimToken.
        # @param [String] name
        #   Identifier. Gemini Enterprise only. The resource name of the SCIM Token.
        #   Format: `locations/`location`/workforcePools/`workforce_pool`/providers/ `
        #   workforce_pool_provider`/scimTenants/`scim_tenant`/tokens/`token``
        # @param [Google::Apis::IamV1::WorkforcePoolProviderScimToken] workforce_pool_provider_scim_token_object
        # @param [String] update_mask
        #   Optional. Gemini Enterprise only. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimToken] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimToken]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_location_workforce_pool_provider_scim_tenant_token(name, workforce_pool_provider_scim_token_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::WorkforcePoolProviderScimToken::Representation
          command.request_object = workforce_pool_provider_scim_token_object
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimToken::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimToken
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gemini Enterprise only. Undeletes a WorkforcePoolProviderScimToken,that was
        # deleted fewer than 30 days ago.
        # @param [String] name
        #   Required. Gemini Enterprise only. The name of the SCIM token to undelete.
        #   Format: `locations/`location`/workforcePools/`workforce_pool`/providers/`
        #   provider`/scimTenants/`scim_tenant`/tokens/`token``
        # @param [Google::Apis::IamV1::UndeleteWorkforcePoolProviderScimTokenRequest] undelete_workforce_pool_provider_scim_token_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkforcePoolProviderScimToken] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkforcePoolProviderScimToken]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workforce_pool_provider_scim_token(name, undelete_workforce_pool_provider_scim_token_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkforcePoolProviderScimTokenRequest::Representation
          command.request_object = undelete_workforce_pool_provider_scim_token_request_object
          command.response_representation = Google::Apis::IamV1::WorkforcePoolProviderScimToken::Representation
          command.response_class = Google::Apis::IamV1::WorkforcePoolProviderScimToken
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a WorkforcePoolSubject. Subject must not already be in a deleted state.
        # A WorkforcePoolSubject is automatically created the first time an external
        # credential is exchanged for a Google Cloud credential using a mapped `google.
        # subject` attribute. There is no endpoint to manually create a
        # WorkforcePoolSubject. For 30 days after a WorkforcePoolSubject is deleted,
        # using the same `google.subject` attribute in token exchanges with Google Cloud
        # STS fails. Call UndeleteWorkforcePoolSubject to undelete a
        # WorkforcePoolSubject that has been deleted, within within 30 days of deleting
        # it. After 30 days, the WorkforcePoolSubject is permanently deleted. At this
        # point, a token exchange with Google Cloud STS that uses the same mapped `
        # google.subject` attribute automatically creates a new WorkforcePoolSubject
        # that is unrelated to the previously deleted WorkforcePoolSubject but has the
        # same `google.subject` value.
        # @param [String] name
        #   Required. The resource name of the WorkforcePoolSubject. Special characters,
        #   like `/` and `:`, must be escaped, because all URLs need to conform to the "
        #   When to Escape and Unescape" section of [RFC3986](https://www.ietf.org/rfc/
        #   rfc2396.txt). Format: `locations/`location`/workforcePools/`workforce_pool_id`/
        #   subjects/`subject_id``
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_location_workforce_pool_subject(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a WorkforcePoolSubject, as long as it was deleted fewer than 30 days
        # ago.
        # @param [String] name
        #   Required. The resource name of the WorkforcePoolSubject. Special characters,
        #   like `/` and `:`, must be escaped, because all URLs need to conform to the "
        #   When to Escape and Unescape" section of [RFC3986](https://www.ietf.org/rfc/
        #   rfc2396.txt). Format: `locations/`location`/workforcePools/`workforce_pool_id`/
        #   subjects/`subject_id``
        # @param [Google::Apis::IamV1::UndeleteWorkforcePoolSubjectRequest] undelete_workforce_pool_subject_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workforce_pool_subject(name, undelete_workforce_pool_subject_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkforcePoolSubjectRequest::Representation
          command.request_object = undelete_workforce_pool_subject_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_location_workforce_pool_subject_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new custom Role.
        # @param [String] parent
        #   The `parent` parameter's value depends on the target resource for the request,
        #   namely [projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.
        #   roles) or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `parent` value format is described
        #   below: * [projects.roles.create](https://cloud.google.com/iam/docs/reference/
        #   rest/v1/projects.roles/create): `projects/`PROJECT_ID``. This method creates
        #   project-level [custom roles](https://cloud.google.com/iam/docs/understanding-
        #   custom-roles). Example request URL: `https://iam.googleapis.com/v1/projects/`
        #   PROJECT_ID`/roles` * [organizations.roles.create](https://cloud.google.com/iam/
        #   docs/reference/rest/v1/organizations.roles/create): `organizations/`
        #   ORGANIZATION_ID``. This method creates organization-level [custom roles](https:
        #   //cloud.google.com/iam/docs/understanding-custom-roles). Example request URL: `
        #   https://iam.googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [Google::Apis::IamV1::CreateRoleRequest] create_role_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_organization_role(parent, create_role_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/roles', options)
          command.request_representation = Google::Apis::IamV1::CreateRoleRequest::Representation
          command.request_object = create_role_request_object
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['parent'] = parent unless parent.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a custom Role. When you delete a custom role, the following changes
        # occur immediately: * You cannot bind a principal to the custom role in an IAM
        # Policy. * Existing bindings to the custom role are not changed, but they have
        # no effect. * By default, the response from ListRoles does not include the
        # custom role. A deleted custom role still counts toward the [custom role limit](
        # https://cloud.google.com/iam/help/limits) until it is permanently deleted. You
        # have 7 days to undelete the custom role. After 7 days, the following changes
        # occur: * The custom role is permanently deleted and cannot be recovered. * If
        # an IAM policy contains a binding to the custom role, the binding is
        # permanently removed. * The custom role no longer counts toward your custom
        # role limit.
        # @param [String] name
        #   The `name` parameter's value depends on the target resource for the request,
        #   namely [projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.
        #   roles) or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `name` value format is described
        #   below: * [projects.roles.delete](https://cloud.google.com/iam/docs/reference/
        #   rest/v1/projects.roles/delete): `projects/`PROJECT_ID`/roles/`CUSTOM_ROLE_ID``.
        #   This method deletes only [custom roles](https://cloud.google.com/iam/docs/
        #   understanding-custom-roles) that have been created at the project level.
        #   Example request URL: `https://iam.googleapis.com/v1/projects/`PROJECT_ID`/
        #   roles/`CUSTOM_ROLE_ID`` * [organizations.roles.delete](https://cloud.google.
        #   com/iam/docs/reference/rest/v1/organizations.roles/delete): `organizations/`
        #   ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID``. This method deletes only [custom
        #   roles](https://cloud.google.com/iam/docs/understanding-custom-roles) that have
        #   been created at the organization level. Example request URL: `https://iam.
        #   googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID`` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [String] etag
        #   Used to perform a consistent read-modify-write.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_organization_role(name, etag: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['name'] = name unless name.nil?
          command.query['etag'] = etag unless etag.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the definition of a Role.
        # @param [String] name
        #   The `name` parameter's value depends on the target resource for the request,
        #   namely [roles](https://cloud.google.com/iam/docs/reference/rest/v1/roles), [
        #   projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.roles),
        #   or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `name` value format is described
        #   below: * [roles.get](https://cloud.google.com/iam/docs/reference/rest/v1/roles/
        #   get): `roles/`ROLE_NAME``. This method returns results from all [predefined
        #   roles](https://cloud.google.com/iam/docs/understanding-roles#predefined_roles)
        #   in IAM. Example request URL: `https://iam.googleapis.com/v1/roles/`ROLE_NAME``
        #   * [projects.roles.get](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   projects.roles/get): `projects/`PROJECT_ID`/roles/`CUSTOM_ROLE_ID``. This
        #   method returns only [custom roles](https://cloud.google.com/iam/docs/
        #   understanding-custom-roles) that have been created at the project level.
        #   Example request URL: `https://iam.googleapis.com/v1/projects/`PROJECT_ID`/
        #   roles/`CUSTOM_ROLE_ID`` * [organizations.roles.get](https://cloud.google.com/
        #   iam/docs/reference/rest/v1/organizations.roles/get): `organizations/`
        #   ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID``. This method returns only [custom
        #   roles](https://cloud.google.com/iam/docs/understanding-custom-roles) that have
        #   been created at the organization level. Example request URL: `https://iam.
        #   googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID`` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_organization_role(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists every predefined Role that IAM supports, or every custom role that is
        # defined for an organization or project.
        # @param [String] parent
        #   The `parent` parameter's value depends on the target resource for the request,
        #   namely [roles](https://cloud.google.com/iam/docs/reference/rest/v1/roles), [
        #   projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.roles),
        #   or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `parent` value format is described
        #   below: * [roles.list](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   roles/list): An empty string. This method doesn't require a resource; it
        #   simply returns all [predefined roles](https://cloud.google.com/iam/docs/
        #   understanding-roles#predefined_roles) in IAM. Example request URL: `https://
        #   iam.googleapis.com/v1/roles` * [projects.roles.list](https://cloud.google.com/
        #   iam/docs/reference/rest/v1/projects.roles/list): `projects/`PROJECT_ID``. This
        #   method lists all project-level [custom roles](https://cloud.google.com/iam/
        #   docs/understanding-custom-roles). Example request URL: `https://iam.googleapis.
        #   com/v1/projects/`PROJECT_ID`/roles` * [organizations.roles.list](https://cloud.
        #   google.com/iam/docs/reference/rest/v1/organizations.roles/list): `
        #   organizations/`ORGANIZATION_ID``. This method lists all organization-level [
        #   custom roles](https://cloud.google.com/iam/docs/understanding-custom-roles).
        #   Example request URL: `https://iam.googleapis.com/v1/organizations/`
        #   ORGANIZATION_ID`/roles` Note: Wildcard (*) values are invalid; you must
        #   specify a complete project ID or organization ID.
        # @param [Fixnum] page_size
        #   Optional limit on the number of roles to include in the response. The default
        #   is 300, and the maximum is 1,000.
        # @param [String] page_token
        #   Optional pagination token returned in an earlier ListRolesResponse.
        # @param [Boolean] show_deleted
        #   Include Roles that have been deleted.
        # @param [String] view
        #   Optional view for the returned Role objects. When `FULL` is specified, the `
        #   includedPermissions` field is returned, which includes a list of all
        #   permissions in the role. The default value is `BASIC`, which does not return
        #   the `includedPermissions` field.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListRolesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListRolesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_organization_roles(parent, page_size: nil, page_token: nil, show_deleted: nil, view: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/roles', options)
          command.response_representation = Google::Apis::IamV1::ListRolesResponse::Representation
          command.response_class = Google::Apis::IamV1::ListRolesResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['view'] = view unless view.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the definition of a custom Role.
        # @param [String] name
        #   The `name` parameter's value depends on the target resource for the request,
        #   namely [projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.
        #   roles) or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `name` value format is described
        #   below: * [projects.roles.patch](https://cloud.google.com/iam/docs/reference/
        #   rest/v1/projects.roles/patch): `projects/`PROJECT_ID`/roles/`CUSTOM_ROLE_ID``.
        #   This method updates only [custom roles](https://cloud.google.com/iam/docs/
        #   understanding-custom-roles) that have been created at the project level.
        #   Example request URL: `https://iam.googleapis.com/v1/projects/`PROJECT_ID`/
        #   roles/`CUSTOM_ROLE_ID`` * [organizations.roles.patch](https://cloud.google.com/
        #   iam/docs/reference/rest/v1/organizations.roles/patch): `organizations/`
        #   ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID``. This method updates only [custom
        #   roles](https://cloud.google.com/iam/docs/understanding-custom-roles) that have
        #   been created at the organization level. Example request URL: `https://iam.
        #   googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID`` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [Google::Apis::IamV1::Role] role_object
        # @param [String] update_mask
        #   A mask describing which fields in the Role have changed.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_organization_role(name, role_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::Role::Representation
          command.request_object = role_object
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a custom Role.
        # @param [String] name
        #   The `name` parameter's value depends on the target resource for the request,
        #   namely [projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.
        #   roles) or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `name` value format is described
        #   below: * [projects.roles.undelete](https://cloud.google.com/iam/docs/reference/
        #   rest/v1/projects.roles/undelete): `projects/`PROJECT_ID`/roles/`CUSTOM_ROLE_ID`
        #   `. This method undeletes only [custom roles](https://cloud.google.com/iam/docs/
        #   understanding-custom-roles) that have been created at the project level.
        #   Example request URL: `https://iam.googleapis.com/v1/projects/`PROJECT_ID`/
        #   roles/`CUSTOM_ROLE_ID`` * [organizations.roles.undelete](https://cloud.google.
        #   com/iam/docs/reference/rest/v1/organizations.roles/undelete): `organizations/`
        #   ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID``. This method undeletes only [custom
        #   roles](https://cloud.google.com/iam/docs/understanding-custom-roles) that have
        #   been created at the organization level. Example request URL: `https://iam.
        #   googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID`` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [Google::Apis::IamV1::UndeleteRoleRequest] undelete_role_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_organization_role(name, undelete_role_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteRoleRequest::Representation
          command.request_object = undelete_role_request_object
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists every permission that you can test on a resource. A permission is
        # testable if you can check whether a principal has that permission on the
        # resource.
        # @param [Google::Apis::IamV1::QueryTestablePermissionsRequest] query_testable_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::QueryTestablePermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::QueryTestablePermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def query_testable_permissions(query_testable_permissions_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/permissions:queryTestablePermissions', options)
          command.request_representation = Google::Apis::IamV1::QueryTestablePermissionsRequest::Representation
          command.request_object = query_testable_permissions_request_object
          command.response_representation = Google::Apis::IamV1::QueryTestablePermissionsResponse::Representation
          command.response_class = Google::Apis::IamV1::QueryTestablePermissionsResponse
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new OauthClient. You cannot reuse the name of a deleted OauthClient
        # until 30 days after deletion.
        # @param [String] parent
        #   Required. The parent resource to create the OauthClient in. The only supported
        #   location is `global`.
        # @param [Google::Apis::IamV1::OauthClient] oauth_client_object
        # @param [String] oauth_client_id
        #   Required. The ID to use for the OauthClient, which becomes the final component
        #   of the resource name. This value should be a string of 6 to 63 lowercase
        #   letters, digits, or hyphens. It must start with a letter, and cannot have a
        #   trailing hyphen. The prefix `gcp-` is reserved for use by Google, and may not
        #   be specified.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::OauthClient] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::OauthClient]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_project_location_oauth_client(parent, oauth_client_object = nil, oauth_client_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/oauthClients', options)
          command.request_representation = Google::Apis::IamV1::OauthClient::Representation
          command.request_object = oauth_client_object
          command.response_representation = Google::Apis::IamV1::OauthClient::Representation
          command.response_class = Google::Apis::IamV1::OauthClient
          command.params['parent'] = parent unless parent.nil?
          command.query['oauthClientId'] = oauth_client_id unless oauth_client_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes an OauthClient. You cannot use a deleted OauthClient. However,
        # deletion does not revoke access tokens that have already been issued. They
        # continue to grant access. Deletion does revoke refresh tokens that have
        # already been issued. They cannot be used to renew an access token. If the
        # OauthClient is undeleted, and the refresh tokens are not expired, they are
        # valid for token exchange again. You can undelete an OauthClient for 30 days.
        # After 30 days, deletion is permanent. You cannot update deleted OauthClients.
        # However, you can view and list them.
        # @param [String] name
        #   Required. The name of the OauthClient to delete. Format: `projects/`project`/
        #   locations/`location`/oauthClients/`oauth_client``.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::OauthClient] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::OauthClient]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_location_oauth_client(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::OauthClient::Representation
          command.response_class = Google::Apis::IamV1::OauthClient
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an individual OauthClient.
        # @param [String] name
        #   Required. The name of the OauthClient to retrieve. Format: `projects/`project`/
        #   locations/`location`/oauthClients/`oauth_client``.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::OauthClient] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::OauthClient]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_oauth_client(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::OauthClient::Representation
          command.response_class = Google::Apis::IamV1::OauthClient
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all non-deleted OauthClients in a project. If `show_deleted` is set to `
        # true`, then deleted OauthClients are also listed.
        # @param [String] parent
        #   Required. The parent to list OauthClients for.
        # @param [Fixnum] page_size
        #   Optional. The maximum number of OauthClients to return. If unspecified, at
        #   most 50 OauthClients will be returned. The maximum value is 100; values above
        #   100 are truncated to 100.
        # @param [String] page_token
        #   Optional. A page token, received from a previous `ListOauthClients` call.
        #   Provide this to retrieve the subsequent page.
        # @param [Boolean] show_deleted
        #   Optional. Whether to return soft-deleted OauthClients.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListOauthClientsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListOauthClientsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_location_oauth_clients(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/oauthClients', options)
          command.response_representation = Google::Apis::IamV1::ListOauthClientsResponse::Representation
          command.response_class = Google::Apis::IamV1::ListOauthClientsResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an existing OauthClient.
        # @param [String] name
        #   Immutable. Identifier. The resource name of the OauthClient. Format:`projects/`
        #   project`/locations/`location`/oauthClients/`oauth_client``.
        # @param [Google::Apis::IamV1::OauthClient] oauth_client_object
        # @param [String] update_mask
        #   Required. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::OauthClient] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::OauthClient]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_project_location_oauth_client(name, oauth_client_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::OauthClient::Representation
          command.request_object = oauth_client_object
          command.response_representation = Google::Apis::IamV1::OauthClient::Representation
          command.response_class = Google::Apis::IamV1::OauthClient
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes an OauthClient, as long as it was deleted fewer than 30 days ago.
        # @param [String] name
        #   Required. The name of the OauthClient to undelete. Format: `projects/`project`/
        #   locations/`location`/oauthClients/`oauth_client``.
        # @param [Google::Apis::IamV1::UndeleteOauthClientRequest] undelete_oauth_client_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::OauthClient] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::OauthClient]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_oauth_client(name, undelete_oauth_client_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteOauthClientRequest::Representation
          command.request_object = undelete_oauth_client_request_object
          command.response_representation = Google::Apis::IamV1::OauthClient::Representation
          command.response_class = Google::Apis::IamV1::OauthClient
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new OauthClientCredential.
        # @param [String] parent
        #   Required. The parent resource to create the OauthClientCredential in.
        # @param [Google::Apis::IamV1::OauthClientCredential] oauth_client_credential_object
        # @param [String] oauth_client_credential_id
        #   Required. The ID to use for the OauthClientCredential, which becomes the final
        #   component of the resource name. This value should be 4-32 characters, and may
        #   contain the characters [a-z0-9-]. The prefix `gcp-` is reserved for use by
        #   Google, and may not be specified.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::OauthClientCredential] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::OauthClientCredential]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_project_location_oauth_client_credential(parent, oauth_client_credential_object = nil, oauth_client_credential_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/credentials', options)
          command.request_representation = Google::Apis::IamV1::OauthClientCredential::Representation
          command.request_object = oauth_client_credential_object
          command.response_representation = Google::Apis::IamV1::OauthClientCredential::Representation
          command.response_class = Google::Apis::IamV1::OauthClientCredential
          command.params['parent'] = parent unless parent.nil?
          command.query['oauthClientCredentialId'] = oauth_client_credential_id unless oauth_client_credential_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes an OauthClientCredential. Before deleting an OauthClientCredential, it
        # should first be disabled.
        # @param [String] name
        #   Required. The name of the OauthClientCredential to delete. Format: `projects/`
        #   project`/locations/`location`/oauthClients/`oauth_client`/credentials/`
        #   credential``.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Empty] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Empty]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_location_oauth_client_credential(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Empty::Representation
          command.response_class = Google::Apis::IamV1::Empty
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an individual OauthClientCredential.
        # @param [String] name
        #   Required. The name of the OauthClientCredential to retrieve. Format: `projects/
        #   `project`/locations/`location`/oauthClients/`oauth_client`/credentials/`
        #   credential``.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::OauthClientCredential] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::OauthClientCredential]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_oauth_client_credential(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::OauthClientCredential::Representation
          command.response_class = Google::Apis::IamV1::OauthClientCredential
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all OauthClientCredentials in an OauthClient.
        # @param [String] parent
        #   Required. The parent to list OauthClientCredentials for.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListOauthClientCredentialsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListOauthClientCredentialsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_location_oauth_client_credentials(parent, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/credentials', options)
          command.response_representation = Google::Apis::IamV1::ListOauthClientCredentialsResponse::Representation
          command.response_class = Google::Apis::IamV1::ListOauthClientCredentialsResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an existing OauthClientCredential.
        # @param [String] name
        #   Immutable. Identifier. The resource name of the OauthClientCredential. Format:
        #   `projects/`project`/locations/`location`/oauthClients/`oauth_client`/
        #   credentials/`credential``
        # @param [Google::Apis::IamV1::OauthClientCredential] oauth_client_credential_object
        # @param [String] update_mask
        #   Required. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::OauthClientCredential] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::OauthClientCredential]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_project_location_oauth_client_credential(name, oauth_client_credential_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::OauthClientCredential::Representation
          command.request_object = oauth_client_credential_object
          command.response_representation = Google::Apis::IamV1::OauthClientCredential::Representation
          command.response_class = Google::Apis::IamV1::OauthClientCredential
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new WorkloadIdentityPool. You cannot reuse the name of a deleted
        # pool until 30 days after deletion.
        # @param [String] parent
        #   Required. The parent resource to create the pool in. The only supported
        #   location is `global`.
        # @param [Google::Apis::IamV1::WorkloadIdentityPool] workload_identity_pool_object
        # @param [String] workload_identity_pool_id
        #   Required. The ID to use for the pool, which becomes the final component of the
        #   resource name. This value should be 4-32 characters, and may contain the
        #   characters [a-z0-9-]. The prefix `gcp-` is reserved for use by Google, and may
        #   not be specified.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_project_location_workload_identity_pool(parent, workload_identity_pool_object = nil, workload_identity_pool_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/workloadIdentityPools', options)
          command.request_representation = Google::Apis::IamV1::WorkloadIdentityPool::Representation
          command.request_object = workload_identity_pool_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['parent'] = parent unless parent.nil?
          command.query['workloadIdentityPoolId'] = workload_identity_pool_id unless workload_identity_pool_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a WorkloadIdentityPool. You cannot use a deleted pool to exchange
        # external credentials for Google Cloud credentials. However, deletion does not
        # revoke credentials that have already been issued. Credentials issued for a
        # deleted pool do not grant access to resources. If the pool is undeleted, and
        # the credentials are not expired, they grant access again. You can undelete a
        # pool for 30 days. After 30 days, deletion is permanent. You cannot update
        # deleted pools. However, you can view and list them.
        # @param [String] name
        #   Required. The name of the pool to delete.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_location_workload_identity_pool(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an individual WorkloadIdentityPool.
        # @param [String] name
        #   Required. The name of the pool to retrieve.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkloadIdentityPool] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkloadIdentityPool]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkloadIdentityPool::Representation
          command.response_class = Google::Apis::IamV1::WorkloadIdentityPool
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the IAM policy of a WorkloadIdentityPool.
        # @param [String] resource
        #   REQUIRED: The resource for which the policy is being requested. See [Resource
        #   names](https://cloud.google.com/apis/design/resource_names) for the
        #   appropriate value for this field.
        # @param [Google::Apis::IamV1::GetIamPolicyRequest] get_iam_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_workload_identity_pool_iam_policy(resource, get_iam_policy_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:getIamPolicy', options)
          command.request_representation = Google::Apis::IamV1::GetIamPolicyRequest::Representation
          command.request_object = get_iam_policy_request_object
          command.response_representation = Google::Apis::IamV1::Policy::Representation
          command.response_class = Google::Apis::IamV1::Policy
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all non-deleted WorkloadIdentityPools in a project. If `show_deleted` is
        # set to `true`, then deleted pools are also listed.
        # @param [String] parent
        #   Required. The parent resource to list pools for.
        # @param [Fixnum] page_size
        #   The maximum number of pools to return. If unspecified, at most 50 pools are
        #   returned. The maximum value is 1000; values above are 1000 truncated to 1000.
        # @param [String] page_token
        #   A page token, received from a previous `ListWorkloadIdentityPools` call.
        #   Provide this to retrieve the subsequent page.
        # @param [Boolean] show_deleted
        #   Whether to return soft-deleted pools.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkloadIdentityPoolsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkloadIdentityPoolsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_location_workload_identity_pools(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/workloadIdentityPools', options)
          command.response_representation = Google::Apis::IamV1::ListWorkloadIdentityPoolsResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkloadIdentityPoolsResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an existing WorkloadIdentityPool.
        # @param [String] name
        #   Output only. The resource name of the pool.
        # @param [Google::Apis::IamV1::WorkloadIdentityPool] workload_identity_pool_object
        # @param [String] update_mask
        #   Required. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_project_location_workload_identity_pool(name, workload_identity_pool_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::WorkloadIdentityPool::Representation
          command.request_object = workload_identity_pool_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the IAM policies on a WorkloadIdentityPool
        # @param [String] resource
        #   REQUIRED: The resource for which the policy is being specified. See [Resource
        #   names](https://cloud.google.com/apis/design/resource_names) for the
        #   appropriate value for this field.
        # @param [Google::Apis::IamV1::SetIamPolicyRequest] set_iam_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_workload_identity_pool_iam_policy(resource, set_iam_policy_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:setIamPolicy', options)
          command.request_representation = Google::Apis::IamV1::SetIamPolicyRequest::Representation
          command.request_object = set_iam_policy_request_object
          command.response_representation = Google::Apis::IamV1::Policy::Representation
          command.response_class = Google::Apis::IamV1::Policy
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the caller's permissions on a WorkloadIdentityPool
        # @param [String] resource
        #   REQUIRED: The resource for which the policy detail is being requested. See [
        #   Resource names](https://cloud.google.com/apis/design/resource_names) for the
        #   appropriate value for this field.
        # @param [Google::Apis::IamV1::TestIamPermissionsRequest] test_iam_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::TestIamPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::TestIamPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_workload_identity_pool_iam_permissions(resource, test_iam_permissions_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:testIamPermissions', options)
          command.request_representation = Google::Apis::IamV1::TestIamPermissionsRequest::Representation
          command.request_object = test_iam_permissions_request_object
          command.response_representation = Google::Apis::IamV1::TestIamPermissionsResponse::Representation
          command.response_class = Google::Apis::IamV1::TestIamPermissionsResponse
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a WorkloadIdentityPool, as long as it was deleted fewer than 30 days
        # ago.
        # @param [String] name
        #   Required. The name of the pool to undelete.
        # @param [Google::Apis::IamV1::UndeleteWorkloadIdentityPoolRequest] undelete_workload_identity_pool_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workload_identity_pool(name, undelete_workload_identity_pool_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkloadIdentityPoolRequest::Representation
          command.request_object = undelete_workload_identity_pool_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new WorkloadIdentityPoolNamespace in a WorkloadIdentityPool.
        # @param [String] parent
        #   Required. The parent resource to create the namespace in. The only supported
        #   location is `global`.
        # @param [Google::Apis::IamV1::WorkloadIdentityPoolNamespace] workload_identity_pool_namespace_object
        # @param [String] workload_identity_pool_namespace_id
        #   Required. The ID to use for the namespace. This value must: * contain at most
        #   63 characters * contain only lowercase alphanumeric characters or `-` * start
        #   with an alphanumeric character * end with an alphanumeric character The prefix
        #   "gcp-" will be reserved for future uses.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_project_location_workload_identity_pool_namespace(parent, workload_identity_pool_namespace_object = nil, workload_identity_pool_namespace_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/namespaces', options)
          command.request_representation = Google::Apis::IamV1::WorkloadIdentityPoolNamespace::Representation
          command.request_object = workload_identity_pool_namespace_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['parent'] = parent unless parent.nil?
          command.query['workloadIdentityPoolNamespaceId'] = workload_identity_pool_namespace_id unless workload_identity_pool_namespace_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a WorkloadIdentityPoolNamespace. You can undelete a namespace for 30
        # days. After 30 days, deletion is permanent.
        # @param [String] name
        #   Required. The name of the namespace to delete.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_location_workload_identity_pool_namespace(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an individual WorkloadIdentityPoolNamespace.
        # @param [String] name
        #   Required. The name of the namespace to retrieve.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkloadIdentityPoolNamespace] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkloadIdentityPoolNamespace]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_namespace(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkloadIdentityPoolNamespace::Representation
          command.response_class = Google::Apis::IamV1::WorkloadIdentityPoolNamespace
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all non-deleted WorkloadIdentityPoolNamespaces in a workload identity
        # pool. If `show_deleted` is set to `true`, then deleted namespaces are also
        # listed.
        # @param [String] parent
        #   Required. The parent resource to list namespaces for.
        # @param [Fixnum] page_size
        #   The maximum number of namespaces to return. If unspecified, at most 50
        #   namespaces are returned. The maximum value is 1000; values above are 1000
        #   truncated to 1000.
        # @param [String] page_token
        #   A page token, received from a previous `ListWorkloadIdentityPoolNamespaces`
        #   call. Provide this to retrieve the subsequent page.
        # @param [Boolean] show_deleted
        #   Whether to return soft-deleted namespaces.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkloadIdentityPoolNamespacesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkloadIdentityPoolNamespacesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_location_workload_identity_pool_namespaces(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/namespaces', options)
          command.response_representation = Google::Apis::IamV1::ListWorkloadIdentityPoolNamespacesResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkloadIdentityPoolNamespacesResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an existing WorkloadIdentityPoolNamespace in a WorkloadIdentityPool.
        # @param [String] name
        #   Output only. The resource name of the namespace.
        # @param [Google::Apis::IamV1::WorkloadIdentityPoolNamespace] workload_identity_pool_namespace_object
        # @param [String] update_mask
        #   Required. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_project_location_workload_identity_pool_namespace(name, workload_identity_pool_namespace_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::WorkloadIdentityPoolNamespace::Representation
          command.request_object = workload_identity_pool_namespace_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a WorkloadIdentityPoolNamespace, as long as it was deleted fewer
        # than 30 days ago.
        # @param [String] name
        #   Required. The name of the namespace to undelete.
        # @param [Google::Apis::IamV1::UndeleteWorkloadIdentityPoolNamespaceRequest] undelete_workload_identity_pool_namespace_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workload_identity_pool_namespace(name, undelete_workload_identity_pool_namespace_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkloadIdentityPoolNamespaceRequest::Representation
          command.request_object = undelete_workload_identity_pool_namespace_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Add an AttestationRule on a WorkloadIdentityPoolManagedIdentity. The total
        # attestation rules after addition must not exceed 50.
        # @param [String] resource
        #   Required. The resource name of the managed identity or namespace resource to
        #   add an attestation rule to.
        # @param [Google::Apis::IamV1::AddAttestationRuleRequest] add_attestation_rule_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_managed_identity_attestation_rule(resource, add_attestation_rule_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:addAttestationRule', options)
          command.request_representation = Google::Apis::IamV1::AddAttestationRuleRequest::Representation
          command.request_object = add_attestation_rule_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new WorkloadIdentityPoolManagedIdentity in a
        # WorkloadIdentityPoolNamespace.
        # @param [String] parent
        #   Required. The parent resource to create the manage identity in. The only
        #   supported location is `global`.
        # @param [Google::Apis::IamV1::WorkloadIdentityPoolManagedIdentity] workload_identity_pool_managed_identity_object
        # @param [String] workload_identity_pool_managed_identity_id
        #   Required. The ID to use for the managed identity. This value must: * contain
        #   at most 63 characters * contain only lowercase alphanumeric characters or `-` *
        #   start with an alphanumeric character * end with an alphanumeric character The
        #   prefix "gcp-" will be reserved for future uses.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_project_location_workload_identity_pool_namespace_managed_identity(parent, workload_identity_pool_managed_identity_object = nil, workload_identity_pool_managed_identity_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/managedIdentities', options)
          command.request_representation = Google::Apis::IamV1::WorkloadIdentityPoolManagedIdentity::Representation
          command.request_object = workload_identity_pool_managed_identity_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['parent'] = parent unless parent.nil?
          command.query['workloadIdentityPoolManagedIdentityId'] = workload_identity_pool_managed_identity_id unless workload_identity_pool_managed_identity_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a WorkloadIdentityPoolManagedIdentity. You can undelete a managed
        # identity for 30 days. After 30 days, deletion is permanent.
        # @param [String] name
        #   Required. The name of the managed identity to delete.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_location_workload_identity_pool_namespace_managed_identity(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an individual WorkloadIdentityPoolManagedIdentity.
        # @param [String] name
        #   Required. The name of the managed identity to retrieve.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkloadIdentityPoolManagedIdentity] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkloadIdentityPoolManagedIdentity]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_namespace_managed_identity(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkloadIdentityPoolManagedIdentity::Representation
          command.response_class = Google::Apis::IamV1::WorkloadIdentityPoolManagedIdentity
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all non-deleted WorkloadIdentityPoolManagedIdentitys in a namespace. If `
        # show_deleted` is set to `true`, then deleted managed identities are also
        # listed.
        # @param [String] parent
        #   Required. The parent resource to list managed identities for.
        # @param [Fixnum] page_size
        #   The maximum number of managed identities to return. If unspecified, at most 50
        #   managed identities are returned. The maximum value is 1000; values above are
        #   1000 truncated to 1000.
        # @param [String] page_token
        #   A page token, received from a previous `
        #   ListWorkloadIdentityPoolManagedIdentities` call. Provide this to retrieve the
        #   subsequent page.
        # @param [Boolean] show_deleted
        #   Whether to return soft-deleted managed identities.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkloadIdentityPoolManagedIdentitiesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkloadIdentityPoolManagedIdentitiesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_location_workload_identity_pool_namespace_managed_identities(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/managedIdentities', options)
          command.response_representation = Google::Apis::IamV1::ListWorkloadIdentityPoolManagedIdentitiesResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkloadIdentityPoolManagedIdentitiesResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # List all AttestationRule on a WorkloadIdentityPoolManagedIdentity.
        # @param [String] resource
        #   Required. The resource name of the managed identity or namespace resource to
        #   list attestation rules of.
        # @param [String] filter
        #   Optional. A query filter. Supports the following function: * `container_ids()`:
        #   Returns only the AttestationRules under the specific container ids. The
        #   function expects a comma-delimited list with only project numbers and must use
        #   the format `projects/`. For example: `container_ids(projects/, projects/,...)`.
        # @param [Fixnum] page_size
        #   Optional. The maximum number of AttestationRules to return. If unspecified, at
        #   most 50 AttestationRules are returned. The maximum value is 100; values above
        #   100 are truncated to 100.
        # @param [String] page_token
        #   Optional. A page token, received from a previous `
        #   ListWorkloadIdentityPoolProviderKeys` call. Provide this to retrieve the
        #   subsequent page.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListAttestationRulesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListAttestationRulesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_location_workload_identity_pool_namespace_managed_identity_attestation_rules(resource, filter: nil, page_size: nil, page_token: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+resource}:listAttestationRules', options)
          command.response_representation = Google::Apis::IamV1::ListAttestationRulesResponse::Representation
          command.response_class = Google::Apis::IamV1::ListAttestationRulesResponse
          command.params['resource'] = resource unless resource.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an existing WorkloadIdentityPoolManagedIdentity in a
        # WorkloadIdentityPoolNamespace.
        # @param [String] name
        #   Output only. The resource name of the managed identity.
        # @param [Google::Apis::IamV1::WorkloadIdentityPoolManagedIdentity] workload_identity_pool_managed_identity_object
        # @param [String] update_mask
        #   Required. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_project_location_workload_identity_pool_namespace_managed_identity(name, workload_identity_pool_managed_identity_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::WorkloadIdentityPoolManagedIdentity::Representation
          command.request_object = workload_identity_pool_managed_identity_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Remove an AttestationRule on a WorkloadIdentityPoolManagedIdentity.
        # @param [String] resource
        #   Required. The resource name of the managed identity or namespace resource to
        #   remove an attestation rule from.
        # @param [Google::Apis::IamV1::RemoveAttestationRuleRequest] remove_attestation_rule_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_managed_identity_attestation_rule(resource, remove_attestation_rule_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:removeAttestationRule', options)
          command.request_representation = Google::Apis::IamV1::RemoveAttestationRuleRequest::Representation
          command.request_object = remove_attestation_rule_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Set all AttestationRule on a WorkloadIdentityPoolManagedIdentity. A maximum of
        # 50 AttestationRules can be set.
        # @param [String] resource
        #   Required. The resource name of the managed identity or namespace resource to
        #   add an attestation rule to.
        # @param [Google::Apis::IamV1::SetAttestationRulesRequest] set_attestation_rules_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_managed_identity_attestation_rules(resource, set_attestation_rules_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:setAttestationRules', options)
          command.request_representation = Google::Apis::IamV1::SetAttestationRulesRequest::Representation
          command.request_object = set_attestation_rules_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a WorkloadIdentityPoolManagedIdentity, as long as it was deleted
        # fewer than 30 days ago.
        # @param [String] name
        #   Required. The name of the managed identity to undelete.
        # @param [Google::Apis::IamV1::UndeleteWorkloadIdentityPoolManagedIdentityRequest] undelete_workload_identity_pool_managed_identity_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workload_identity_pool_managed_identity(name, undelete_workload_identity_pool_managed_identity_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkloadIdentityPoolManagedIdentityRequest::Representation
          command.request_object = undelete_workload_identity_pool_managed_identity_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_namespace_managed_identity_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_namespace_managed_identity_workload_source_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_namespace_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new WorkloadIdentityPoolProvider in a WorkloadIdentityPool. You
        # cannot reuse the name of a deleted provider until 30 days after deletion.
        # @param [String] parent
        #   Required. The pool to create this provider in.
        # @param [Google::Apis::IamV1::WorkloadIdentityPoolProvider] workload_identity_pool_provider_object
        # @param [String] workload_identity_pool_provider_id
        #   Required. The ID for the provider, which becomes the final component of the
        #   resource name. This value must be 4-32 characters, and may contain the
        #   characters [a-z0-9-]. The prefix `gcp-` is reserved for use by Google, and may
        #   not be specified.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_project_location_workload_identity_pool_provider(parent, workload_identity_pool_provider_object = nil, workload_identity_pool_provider_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/providers', options)
          command.request_representation = Google::Apis::IamV1::WorkloadIdentityPoolProvider::Representation
          command.request_object = workload_identity_pool_provider_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['parent'] = parent unless parent.nil?
          command.query['workloadIdentityPoolProviderId'] = workload_identity_pool_provider_id unless workload_identity_pool_provider_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a WorkloadIdentityPoolProvider. Deleting a provider does not revoke
        # credentials that have already been issued; they continue to grant access. You
        # can undelete a provider for 30 days. After 30 days, deletion is permanent. You
        # cannot update deleted providers. However, you can view and list them.
        # @param [String] name
        #   Required. The name of the provider to delete.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_location_workload_identity_pool_provider(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an individual WorkloadIdentityPoolProvider.
        # @param [String] name
        #   Required. The name of the provider to retrieve.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkloadIdentityPoolProvider] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkloadIdentityPoolProvider]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_provider(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkloadIdentityPoolProvider::Representation
          command.response_class = Google::Apis::IamV1::WorkloadIdentityPoolProvider
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all non-deleted WorkloadIdentityPoolProviders in a WorkloadIdentityPool.
        # If `show_deleted` is set to `true`, then deleted providers are also listed.
        # @param [String] parent
        #   Required. The pool to list providers for.
        # @param [Fixnum] page_size
        #   The maximum number of providers to return. If unspecified, at most 50
        #   providers are returned. The maximum value is 100; values above 100 are
        #   truncated to 100.
        # @param [String] page_token
        #   A page token, received from a previous `ListWorkloadIdentityPoolProviders`
        #   call. Provide this to retrieve the subsequent page.
        # @param [Boolean] show_deleted
        #   Whether to return soft-deleted providers.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkloadIdentityPoolProvidersResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkloadIdentityPoolProvidersResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_location_workload_identity_pool_providers(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/providers', options)
          command.response_representation = Google::Apis::IamV1::ListWorkloadIdentityPoolProvidersResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkloadIdentityPoolProvidersResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an existing WorkloadIdentityPoolProvider.
        # @param [String] name
        #   Output only. The resource name of the provider.
        # @param [Google::Apis::IamV1::WorkloadIdentityPoolProvider] workload_identity_pool_provider_object
        # @param [String] update_mask
        #   Required. The list of fields to update.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_project_location_workload_identity_pool_provider(name, workload_identity_pool_provider_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::WorkloadIdentityPoolProvider::Representation
          command.request_object = workload_identity_pool_provider_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a WorkloadIdentityPoolProvider, as long as it was deleted fewer than
        # 30 days ago.
        # @param [String] name
        #   Required. The name of the provider to undelete.
        # @param [Google::Apis::IamV1::UndeleteWorkloadIdentityPoolProviderRequest] undelete_workload_identity_pool_provider_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workload_identity_pool_provider(name, undelete_workload_identity_pool_provider_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkloadIdentityPoolProviderRequest::Representation
          command.request_object = undelete_workload_identity_pool_provider_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Create a new WorkloadIdentityPoolProviderKey in a WorkloadIdentityPoolProvider.
        # @param [String] parent
        #   Required. The parent provider resource to create the key in.
        # @param [Google::Apis::IamV1::WorkloadIdentityPoolProviderKey] workload_identity_pool_provider_key_object
        # @param [String] workload_identity_pool_provider_key_id
        #   Required. The ID to use for the key, which becomes the final component of the
        #   resource name. This value should be 4-32 characters, and may contain the
        #   characters [a-z0-9-].
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_project_location_workload_identity_pool_provider_key(parent, workload_identity_pool_provider_key_object = nil, workload_identity_pool_provider_key_id: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/keys', options)
          command.request_representation = Google::Apis::IamV1::WorkloadIdentityPoolProviderKey::Representation
          command.request_object = workload_identity_pool_provider_key_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['parent'] = parent unless parent.nil?
          command.query['workloadIdentityPoolProviderKeyId'] = workload_identity_pool_provider_key_id unless workload_identity_pool_provider_key_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes an WorkloadIdentityPoolProviderKey. You can undelete a key for 30 days.
        # After 30 days, deletion is permanent.
        # @param [String] name
        #   Required. The name of the encryption key to delete.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_location_workload_identity_pool_provider_key(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an individual WorkloadIdentityPoolProviderKey.
        # @param [String] name
        #   Required. The name of the key to retrieve.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::WorkloadIdentityPoolProviderKey] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::WorkloadIdentityPoolProviderKey]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_provider_key(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::WorkloadIdentityPoolProviderKey::Representation
          command.response_class = Google::Apis::IamV1::WorkloadIdentityPoolProviderKey
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all non-deleted WorkloadIdentityPoolProviderKeys in a project. If
        # show_deleted is set to `true`, then deleted pools are also listed.
        # @param [String] parent
        #   Required. The parent provider resource to list encryption keys for.
        # @param [Fixnum] page_size
        #   The maximum number of keys to return. If unspecified, all keys are returned.
        #   The maximum value is 10; values above 10 are truncated to 10.
        # @param [String] page_token
        #   A page token, received from a previous `ListWorkloadIdentityPoolProviderKeys`
        #   call. Provide this to retrieve the subsequent page.
        # @param [Boolean] show_deleted
        #   Whether to return soft deleted resources as well.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListWorkloadIdentityPoolProviderKeysResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListWorkloadIdentityPoolProviderKeysResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_location_workload_identity_pool_provider_keys(parent, page_size: nil, page_token: nil, show_deleted: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/keys', options)
          command.response_representation = Google::Apis::IamV1::ListWorkloadIdentityPoolProviderKeysResponse::Representation
          command.response_class = Google::Apis::IamV1::ListWorkloadIdentityPoolProviderKeysResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes an WorkloadIdentityPoolProviderKey, as long as it was deleted fewer
        # than 30 days ago.
        # @param [String] name
        #   Required. The name of the encryption key to undelete.
        # @param [Google::Apis::IamV1::UndeleteWorkloadIdentityPoolProviderKeyRequest] undelete_workload_identity_pool_provider_key_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_workload_identity_pool_provider_key(name, undelete_workload_identity_pool_provider_key_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteWorkloadIdentityPoolProviderKeyRequest::Representation
          command.request_object = undelete_workload_identity_pool_provider_key_request_object
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_provider_key_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the latest state of a long-running operation. Clients can use this method
        # to poll the operation result at intervals as recommended by the API service.
        # @param [String] name
        #   The name of the operation resource.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_location_workload_identity_pool_provider_operation(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Operation::Representation
          command.response_class = Google::Apis::IamV1::Operation
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new custom Role.
        # @param [String] parent
        #   The `parent` parameter's value depends on the target resource for the request,
        #   namely [projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.
        #   roles) or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `parent` value format is described
        #   below: * [projects.roles.create](https://cloud.google.com/iam/docs/reference/
        #   rest/v1/projects.roles/create): `projects/`PROJECT_ID``. This method creates
        #   project-level [custom roles](https://cloud.google.com/iam/docs/understanding-
        #   custom-roles). Example request URL: `https://iam.googleapis.com/v1/projects/`
        #   PROJECT_ID`/roles` * [organizations.roles.create](https://cloud.google.com/iam/
        #   docs/reference/rest/v1/organizations.roles/create): `organizations/`
        #   ORGANIZATION_ID``. This method creates organization-level [custom roles](https:
        #   //cloud.google.com/iam/docs/understanding-custom-roles). Example request URL: `
        #   https://iam.googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [Google::Apis::IamV1::CreateRoleRequest] create_role_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_project_role(parent, create_role_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+parent}/roles', options)
          command.request_representation = Google::Apis::IamV1::CreateRoleRequest::Representation
          command.request_object = create_role_request_object
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['parent'] = parent unless parent.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a custom Role. When you delete a custom role, the following changes
        # occur immediately: * You cannot bind a principal to the custom role in an IAM
        # Policy. * Existing bindings to the custom role are not changed, but they have
        # no effect. * By default, the response from ListRoles does not include the
        # custom role. A deleted custom role still counts toward the [custom role limit](
        # https://cloud.google.com/iam/help/limits) until it is permanently deleted. You
        # have 7 days to undelete the custom role. After 7 days, the following changes
        # occur: * The custom role is permanently deleted and cannot be recovered. * If
        # an IAM policy contains a binding to the custom role, the binding is
        # permanently removed. * The custom role no longer counts toward your custom
        # role limit.
        # @param [String] name
        #   The `name` parameter's value depends on the target resource for the request,
        #   namely [projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.
        #   roles) or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `name` value format is described
        #   below: * [projects.roles.delete](https://cloud.google.com/iam/docs/reference/
        #   rest/v1/projects.roles/delete): `projects/`PROJECT_ID`/roles/`CUSTOM_ROLE_ID``.
        #   This method deletes only [custom roles](https://cloud.google.com/iam/docs/
        #   understanding-custom-roles) that have been created at the project level.
        #   Example request URL: `https://iam.googleapis.com/v1/projects/`PROJECT_ID`/
        #   roles/`CUSTOM_ROLE_ID`` * [organizations.roles.delete](https://cloud.google.
        #   com/iam/docs/reference/rest/v1/organizations.roles/delete): `organizations/`
        #   ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID``. This method deletes only [custom
        #   roles](https://cloud.google.com/iam/docs/understanding-custom-roles) that have
        #   been created at the organization level. Example request URL: `https://iam.
        #   googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID`` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [String] etag
        #   Used to perform a consistent read-modify-write.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_role(name, etag: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['name'] = name unless name.nil?
          command.query['etag'] = etag unless etag.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the definition of a Role.
        # @param [String] name
        #   The `name` parameter's value depends on the target resource for the request,
        #   namely [roles](https://cloud.google.com/iam/docs/reference/rest/v1/roles), [
        #   projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.roles),
        #   or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `name` value format is described
        #   below: * [roles.get](https://cloud.google.com/iam/docs/reference/rest/v1/roles/
        #   get): `roles/`ROLE_NAME``. This method returns results from all [predefined
        #   roles](https://cloud.google.com/iam/docs/understanding-roles#predefined_roles)
        #   in IAM. Example request URL: `https://iam.googleapis.com/v1/roles/`ROLE_NAME``
        #   * [projects.roles.get](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   projects.roles/get): `projects/`PROJECT_ID`/roles/`CUSTOM_ROLE_ID``. This
        #   method returns only [custom roles](https://cloud.google.com/iam/docs/
        #   understanding-custom-roles) that have been created at the project level.
        #   Example request URL: `https://iam.googleapis.com/v1/projects/`PROJECT_ID`/
        #   roles/`CUSTOM_ROLE_ID`` * [organizations.roles.get](https://cloud.google.com/
        #   iam/docs/reference/rest/v1/organizations.roles/get): `organizations/`
        #   ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID``. This method returns only [custom
        #   roles](https://cloud.google.com/iam/docs/understanding-custom-roles) that have
        #   been created at the organization level. Example request URL: `https://iam.
        #   googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID`` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_role(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists every predefined Role that IAM supports, or every custom role that is
        # defined for an organization or project.
        # @param [String] parent
        #   The `parent` parameter's value depends on the target resource for the request,
        #   namely [roles](https://cloud.google.com/iam/docs/reference/rest/v1/roles), [
        #   projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.roles),
        #   or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `parent` value format is described
        #   below: * [roles.list](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   roles/list): An empty string. This method doesn't require a resource; it
        #   simply returns all [predefined roles](https://cloud.google.com/iam/docs/
        #   understanding-roles#predefined_roles) in IAM. Example request URL: `https://
        #   iam.googleapis.com/v1/roles` * [projects.roles.list](https://cloud.google.com/
        #   iam/docs/reference/rest/v1/projects.roles/list): `projects/`PROJECT_ID``. This
        #   method lists all project-level [custom roles](https://cloud.google.com/iam/
        #   docs/understanding-custom-roles). Example request URL: `https://iam.googleapis.
        #   com/v1/projects/`PROJECT_ID`/roles` * [organizations.roles.list](https://cloud.
        #   google.com/iam/docs/reference/rest/v1/organizations.roles/list): `
        #   organizations/`ORGANIZATION_ID``. This method lists all organization-level [
        #   custom roles](https://cloud.google.com/iam/docs/understanding-custom-roles).
        #   Example request URL: `https://iam.googleapis.com/v1/organizations/`
        #   ORGANIZATION_ID`/roles` Note: Wildcard (*) values are invalid; you must
        #   specify a complete project ID or organization ID.
        # @param [Fixnum] page_size
        #   Optional limit on the number of roles to include in the response. The default
        #   is 300, and the maximum is 1,000.
        # @param [String] page_token
        #   Optional pagination token returned in an earlier ListRolesResponse.
        # @param [Boolean] show_deleted
        #   Include Roles that have been deleted.
        # @param [String] view
        #   Optional view for the returned Role objects. When `FULL` is specified, the `
        #   includedPermissions` field is returned, which includes a list of all
        #   permissions in the role. The default value is `BASIC`, which does not return
        #   the `includedPermissions` field.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListRolesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListRolesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_roles(parent, page_size: nil, page_token: nil, show_deleted: nil, view: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+parent}/roles', options)
          command.response_representation = Google::Apis::IamV1::ListRolesResponse::Representation
          command.response_class = Google::Apis::IamV1::ListRolesResponse
          command.params['parent'] = parent unless parent.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['view'] = view unless view.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the definition of a custom Role.
        # @param [String] name
        #   The `name` parameter's value depends on the target resource for the request,
        #   namely [projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.
        #   roles) or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `name` value format is described
        #   below: * [projects.roles.patch](https://cloud.google.com/iam/docs/reference/
        #   rest/v1/projects.roles/patch): `projects/`PROJECT_ID`/roles/`CUSTOM_ROLE_ID``.
        #   This method updates only [custom roles](https://cloud.google.com/iam/docs/
        #   understanding-custom-roles) that have been created at the project level.
        #   Example request URL: `https://iam.googleapis.com/v1/projects/`PROJECT_ID`/
        #   roles/`CUSTOM_ROLE_ID`` * [organizations.roles.patch](https://cloud.google.com/
        #   iam/docs/reference/rest/v1/organizations.roles/patch): `organizations/`
        #   ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID``. This method updates only [custom
        #   roles](https://cloud.google.com/iam/docs/understanding-custom-roles) that have
        #   been created at the organization level. Example request URL: `https://iam.
        #   googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID`` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [Google::Apis::IamV1::Role] role_object
        # @param [String] update_mask
        #   A mask describing which fields in the Role have changed.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_project_role(name, role_object = nil, update_mask: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::Role::Representation
          command.request_object = role_object
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['name'] = name unless name.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Undeletes a custom Role.
        # @param [String] name
        #   The `name` parameter's value depends on the target resource for the request,
        #   namely [projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.
        #   roles) or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `name` value format is described
        #   below: * [projects.roles.undelete](https://cloud.google.com/iam/docs/reference/
        #   rest/v1/projects.roles/undelete): `projects/`PROJECT_ID`/roles/`CUSTOM_ROLE_ID`
        #   `. This method undeletes only [custom roles](https://cloud.google.com/iam/docs/
        #   understanding-custom-roles) that have been created at the project level.
        #   Example request URL: `https://iam.googleapis.com/v1/projects/`PROJECT_ID`/
        #   roles/`CUSTOM_ROLE_ID`` * [organizations.roles.undelete](https://cloud.google.
        #   com/iam/docs/reference/rest/v1/organizations.roles/undelete): `organizations/`
        #   ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID``. This method undeletes only [custom
        #   roles](https://cloud.google.com/iam/docs/understanding-custom-roles) that have
        #   been created at the organization level. Example request URL: `https://iam.
        #   googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID`` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [Google::Apis::IamV1::UndeleteRoleRequest] undelete_role_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_project_role(name, undelete_role_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteRoleRequest::Representation
          command.request_object = undelete_role_request_object
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a ServiceAccount.
        # @param [String] name
        #   Required. The resource name of the project associated with the service
        #   accounts, such as `projects/my-project-123`.
        # @param [Google::Apis::IamV1::CreateServiceAccountRequest] create_service_account_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ServiceAccount] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ServiceAccount]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_service_account(name, create_service_account_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}/serviceAccounts', options)
          command.request_representation = Google::Apis::IamV1::CreateServiceAccountRequest::Representation
          command.request_object = create_service_account_request_object
          command.response_representation = Google::Apis::IamV1::ServiceAccount::Representation
          command.response_class = Google::Apis::IamV1::ServiceAccount
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a ServiceAccount. **Warning:** After you delete a service account, you
        # might not be able to undelete it. If you know that you need to re-enable the
        # service account in the future, use DisableServiceAccount instead. If you
        # delete a service account, IAM permanently removes the service account 30 days
        # later. Google Cloud cannot recover the service account after it is permanently
        # removed, even if you file a support request. To help avoid unplanned outages,
        # we recommend that you disable the service account before you delete it. Use
        # DisableServiceAccount to disable the service account, then wait at least 24
        # hours and watch for unintended consequences. If there are no unintended
        # consequences, you can delete the service account.
        # @param [String] name
        #   Required. The resource name of the service account. Use one of the following
        #   formats: * `projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/
        #   `PROJECT_ID`/serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-
        #   ` wildcard character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Empty] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Empty]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_service_account(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Empty::Representation
          command.response_class = Google::Apis::IamV1::Empty
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Disables a ServiceAccount immediately. If an application uses the service
        # account to authenticate, that application can no longer call Google APIs or
        # access Google Cloud resources. Existing access tokens for the service account
        # are rejected, and requests for new access tokens will fail. To re-enable the
        # service account, use EnableServiceAccount. After you re-enable the service
        # account, its existing access tokens will be accepted, and you can request new
        # access tokens. To help avoid unplanned outages, we recommend that you disable
        # the service account before you delete it. Use this method to disable the
        # service account, then wait at least 24 hours and watch for unintended
        # consequences. If there are no unintended consequences, you can delete the
        # service account with DeleteServiceAccount.
        # @param [String] name
        #   The resource name of the service account. Use one of the following formats: * `
        #   projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/`PROJECT_ID`
        #   /serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-` wildcard
        #   character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [Google::Apis::IamV1::DisableServiceAccountRequest] disable_service_account_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Empty] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Empty]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def disable_service_account(name, disable_service_account_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:disable', options)
          command.request_representation = Google::Apis::IamV1::DisableServiceAccountRequest::Representation
          command.request_object = disable_service_account_request_object
          command.response_representation = Google::Apis::IamV1::Empty::Representation
          command.response_class = Google::Apis::IamV1::Empty
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Enables a ServiceAccount that was disabled by DisableServiceAccount. If the
        # service account is already enabled, then this method has no effect. If the
        # service account was disabled by other means—for example, if Google disabled
        # the service account because it was compromised—you cannot use this method to
        # enable the service account.
        # @param [String] name
        #   The resource name of the service account. Use one of the following formats: * `
        #   projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/`PROJECT_ID`
        #   /serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-` wildcard
        #   character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [Google::Apis::IamV1::EnableServiceAccountRequest] enable_service_account_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Empty] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Empty]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def enable_service_account(name, enable_service_account_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:enable', options)
          command.request_representation = Google::Apis::IamV1::EnableServiceAccountRequest::Representation
          command.request_object = enable_service_account_request_object
          command.response_representation = Google::Apis::IamV1::Empty::Representation
          command.response_class = Google::Apis::IamV1::Empty
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets a ServiceAccount.
        # @param [String] name
        #   Required. The resource name of the service account. Use one of the following
        #   formats: * `projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/
        #   `PROJECT_ID`/serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-
        #   ` wildcard character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ServiceAccount] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ServiceAccount]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_service_account(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::ServiceAccount::Representation
          command.response_class = Google::Apis::IamV1::ServiceAccount
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the IAM policy that is attached to a ServiceAccount. This IAM policy
        # specifies which principals have access to the service account. This method
        # does not tell you whether the service account has been granted any roles on
        # other resources. To check whether a service account has role grants on a
        # resource, use the `getIamPolicy` method for that resource. For example, to
        # view the role grants for a project, call the Resource Manager API's [projects.
        # getIamPolicy](https://cloud.google.com/resource-manager/reference/rest/v1/
        # projects/getIamPolicy) method.
        # @param [String] resource
        #   REQUIRED: The resource for which the policy is being requested. See [Resource
        #   names](https://cloud.google.com/apis/design/resource_names) for the
        #   appropriate value for this field.
        # @param [Fixnum] options_requested_policy_version
        #   Optional. The maximum policy version that will be used to format the policy.
        #   Valid values are 0, 1, and 3. Requests specifying an invalid value will be
        #   rejected. Requests for policies with any conditional role bindings must
        #   specify version 3. Policies with no conditional role bindings may specify any
        #   valid value or leave the field unset. The policy in the response might use the
        #   policy version that you specified, or it might use a lower policy version. For
        #   example, if you specify version 3, but the policy has no conditional role
        #   bindings, the response uses version 1. To learn which resources support
        #   conditions in their IAM policies, see the [IAM documentation](https://cloud.
        #   google.com/iam/help/conditions/resource-policies).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_service_account_iam_policy(resource, options_requested_policy_version: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:getIamPolicy', options)
          command.response_representation = Google::Apis::IamV1::Policy::Representation
          command.response_class = Google::Apis::IamV1::Policy
          command.params['resource'] = resource unless resource.nil?
          command.query['options.requestedPolicyVersion'] = options_requested_policy_version unless options_requested_policy_version.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists every ServiceAccount that belongs to a specific project.
        # @param [String] name
        #   Required. The resource name of the project associated with the service
        #   accounts, such as `projects/my-project-123`.
        # @param [Fixnum] page_size
        #   Optional limit on the number of service accounts to include in the response.
        #   Further accounts can subsequently be obtained by including the
        #   ListServiceAccountsResponse.next_page_token in a subsequent request. The
        #   default is 20, and the maximum is 100.
        # @param [String] page_token
        #   Optional pagination token returned in an earlier ListServiceAccountsResponse.
        #   next_page_token.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListServiceAccountsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListServiceAccountsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_service_accounts(name, page_size: nil, page_token: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}/serviceAccounts', options)
          command.response_representation = Google::Apis::IamV1::ListServiceAccountsResponse::Representation
          command.response_class = Google::Apis::IamV1::ListServiceAccountsResponse
          command.params['name'] = name unless name.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches a ServiceAccount.
        # @param [String] name
        #   The resource name of the service account. Use one of the following formats: * `
        #   projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/`PROJECT_ID`
        #   /serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-` wildcard
        #   character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [Google::Apis::IamV1::PatchServiceAccountRequest] patch_service_account_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ServiceAccount] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ServiceAccount]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_service_account(name, patch_service_account_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:patch, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::PatchServiceAccountRequest::Representation
          command.request_object = patch_service_account_request_object
          command.response_representation = Google::Apis::IamV1::ServiceAccount::Representation
          command.response_class = Google::Apis::IamV1::ServiceAccount
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the IAM policy that is attached to a ServiceAccount. Use this method to
        # grant or revoke access to the service account. For example, you could grant a
        # principal the ability to impersonate the service account. This method does not
        # enable the service account to access other resources. To grant roles to a
        # service account on a resource, follow these steps: 1. Call the resource's `
        # getIamPolicy` method to get its current IAM policy. 2. Edit the policy so that
        # it binds the service account to an IAM role for the resource. 3. Call the
        # resource's `setIamPolicy` method to update its IAM policy. For detailed
        # instructions, see [Manage access to project, folders, and organizations](https:
        # //cloud.google.com/iam/help/service-accounts/granting-access-to-service-
        # accounts) or [Manage access to other resources](https://cloud.google.com/iam/
        # help/access/manage-other-resources).
        # @param [String] resource
        #   REQUIRED: The resource for which the policy is being specified. See [Resource
        #   names](https://cloud.google.com/apis/design/resource_names) for the
        #   appropriate value for this field.
        # @param [Google::Apis::IamV1::SetIamPolicyRequest] set_iam_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_service_account_iam_policy(resource, set_iam_policy_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:setIamPolicy', options)
          command.request_representation = Google::Apis::IamV1::SetIamPolicyRequest::Representation
          command.request_object = set_iam_policy_request_object
          command.response_representation = Google::Apis::IamV1::Policy::Representation
          command.response_class = Google::Apis::IamV1::Policy
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Signs a blob using the system-managed private key for a ServiceAccount.
        # @param [String] name
        #   Required. Deprecated. [Migrate to Service Account Credentials API](https://
        #   cloud.google.com/iam/help/credentials/migrate-api). The resource name of the
        #   service account. Use one of the following formats: * `projects/`PROJECT_ID`/
        #   serviceAccounts/`EMAIL_ADDRESS`` * `projects/`PROJECT_ID`/serviceAccounts/`
        #   UNIQUE_ID`` As an alternative, you can use the `-` wildcard character instead
        #   of the project ID: * `projects/-/serviceAccounts/`EMAIL_ADDRESS`` * `projects/-
        #   /serviceAccounts/`UNIQUE_ID`` When possible, avoid using the `-` wildcard
        #   character, because it can cause response messages to contain misleading error
        #   codes. For example, if you try to access the service account `projects/-/
        #   serviceAccounts/fake@example.com`, which does not exist, the response contains
        #   an HTTP `403 Forbidden` error instead of a `404 Not Found` error.
        # @param [Google::Apis::IamV1::SignBlobRequest] sign_blob_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::SignBlobResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::SignBlobResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def sign_service_account_blob(name, sign_blob_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:signBlob', options)
          command.request_representation = Google::Apis::IamV1::SignBlobRequest::Representation
          command.request_object = sign_blob_request_object
          command.response_representation = Google::Apis::IamV1::SignBlobResponse::Representation
          command.response_class = Google::Apis::IamV1::SignBlobResponse
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Signs a JSON Web Token (JWT) using the system-managed private key for a
        # ServiceAccount.
        # @param [String] name
        #   Required. Deprecated. [Migrate to Service Account Credentials API](https://
        #   cloud.google.com/iam/help/credentials/migrate-api). The resource name of the
        #   service account. Use one of the following formats: * `projects/`PROJECT_ID`/
        #   serviceAccounts/`EMAIL_ADDRESS`` * `projects/`PROJECT_ID`/serviceAccounts/`
        #   UNIQUE_ID`` As an alternative, you can use the `-` wildcard character instead
        #   of the project ID: * `projects/-/serviceAccounts/`EMAIL_ADDRESS`` * `projects/-
        #   /serviceAccounts/`UNIQUE_ID`` When possible, avoid using the `-` wildcard
        #   character, because it can cause response messages to contain misleading error
        #   codes. For example, if you try to access the service account `projects/-/
        #   serviceAccounts/fake@example.com`, which does not exist, the response contains
        #   an HTTP `403 Forbidden` error instead of a `404 Not Found` error.
        # @param [Google::Apis::IamV1::SignJwtRequest] sign_jwt_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::SignJwtResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::SignJwtResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def sign_service_account_jwt(name, sign_jwt_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:signJwt', options)
          command.request_representation = Google::Apis::IamV1::SignJwtRequest::Representation
          command.request_object = sign_jwt_request_object
          command.response_representation = Google::Apis::IamV1::SignJwtResponse::Representation
          command.response_class = Google::Apis::IamV1::SignJwtResponse
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Tests whether the caller has the specified permissions on a ServiceAccount.
        # @param [String] resource
        #   REQUIRED: The resource for which the policy detail is being requested. See [
        #   Resource names](https://cloud.google.com/apis/design/resource_names) for the
        #   appropriate value for this field.
        # @param [Google::Apis::IamV1::TestIamPermissionsRequest] test_iam_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::TestIamPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::TestIamPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_service_account_iam_permissions(resource, test_iam_permissions_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+resource}:testIamPermissions', options)
          command.request_representation = Google::Apis::IamV1::TestIamPermissionsRequest::Representation
          command.request_object = test_iam_permissions_request_object
          command.response_representation = Google::Apis::IamV1::TestIamPermissionsResponse::Representation
          command.response_class = Google::Apis::IamV1::TestIamPermissionsResponse
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Restores a deleted ServiceAccount. **Important:** It is not always possible to
        # restore a deleted service account. Use this method only as a last resort.
        # After you delete a service account, IAM permanently removes the service
        # account 30 days later. There is no way to restore a deleted service account
        # that has been permanently removed.
        # @param [String] name
        #   The resource name of the service account. Use one of the following formats: * `
        #   projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/`PROJECT_ID`
        #   /serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-` wildcard
        #   character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [Google::Apis::IamV1::UndeleteServiceAccountRequest] undelete_service_account_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::UndeleteServiceAccountResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::UndeleteServiceAccountResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def undelete_service_account(name, undelete_service_account_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:undelete', options)
          command.request_representation = Google::Apis::IamV1::UndeleteServiceAccountRequest::Representation
          command.request_object = undelete_service_account_request_object
          command.response_representation = Google::Apis::IamV1::UndeleteServiceAccountResponse::Representation
          command.response_class = Google::Apis::IamV1::UndeleteServiceAccountResponse
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # **Note:** We are in the process of deprecating this method. Use
        # PatchServiceAccount instead. Updates a ServiceAccount. You can update only the
        # `display_name` field.
        # @param [String] name
        #   The resource name of the service account. Use one of the following formats: * `
        #   projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/`PROJECT_ID`
        #   /serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-` wildcard
        #   character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [Google::Apis::IamV1::ServiceAccount] service_account_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ServiceAccount] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ServiceAccount]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_project_service_account(name, service_account_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:put, 'v1/{+name}', options)
          command.request_representation = Google::Apis::IamV1::ServiceAccount::Representation
          command.request_object = service_account_object
          command.response_representation = Google::Apis::IamV1::ServiceAccount::Representation
          command.response_class = Google::Apis::IamV1::ServiceAccount
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a ServiceAccountKey.
        # @param [String] name
        #   Required. The resource name of the service account. Use one of the following
        #   formats: * `projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/
        #   `PROJECT_ID`/serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-
        #   ` wildcard character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [Google::Apis::IamV1::CreateServiceAccountKeyRequest] create_service_account_key_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ServiceAccountKey] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ServiceAccountKey]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_service_account_key(name, create_service_account_key_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}/keys', options)
          command.request_representation = Google::Apis::IamV1::CreateServiceAccountKeyRequest::Representation
          command.request_object = create_service_account_key_request_object
          command.response_representation = Google::Apis::IamV1::ServiceAccountKey::Representation
          command.response_class = Google::Apis::IamV1::ServiceAccountKey
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a ServiceAccountKey. Deleting a service account key does not revoke
        # short-lived credentials that have been issued based on the service account key.
        # @param [String] name
        #   Required. The resource name of the service account key. Use one of the
        #   following formats: * `projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`/
        #   keys/`KEY_ID`` * `projects/`PROJECT_ID`/serviceAccounts/`UNIQUE_ID`/keys/`
        #   KEY_ID`` As an alternative, you can use the `-` wildcard character instead of
        #   the project ID: * `projects/-/serviceAccounts/`EMAIL_ADDRESS`/keys/`KEY_ID`` *
        #   `projects/-/serviceAccounts/`UNIQUE_ID`/keys/`KEY_ID`` When possible, avoid
        #   using the `-` wildcard character, because it can cause response messages to
        #   contain misleading error codes. For example, if you try to access the service
        #   account key `projects/-/serviceAccounts/fake@example.com/keys/fake-key`, which
        #   does not exist, the response contains an HTTP `403 Forbidden` error instead of
        #   a `404 Not Found` error.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Empty] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Empty]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_project_service_account_key(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:delete, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Empty::Representation
          command.response_class = Google::Apis::IamV1::Empty
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Disable a ServiceAccountKey. A disabled service account key can be re-enabled
        # with EnableServiceAccountKey.
        # @param [String] name
        #   Required. The resource name of the service account key. Use one of the
        #   following formats: * `projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`/
        #   keys/`KEY_ID`` * `projects/`PROJECT_ID`/serviceAccounts/`UNIQUE_ID`/keys/`
        #   KEY_ID`` As an alternative, you can use the `-` wildcard character instead of
        #   the project ID: * `projects/-/serviceAccounts/`EMAIL_ADDRESS`/keys/`KEY_ID`` *
        #   `projects/-/serviceAccounts/`UNIQUE_ID`/keys/`KEY_ID`` When possible, avoid
        #   using the `-` wildcard character, because it can cause response messages to
        #   contain misleading error codes. For example, if you try to access the service
        #   account key `projects/-/serviceAccounts/fake@example.com/keys/fake-key`, which
        #   does not exist, the response contains an HTTP `403 Forbidden` error instead of
        #   a `404 Not Found` error.
        # @param [Google::Apis::IamV1::DisableServiceAccountKeyRequest] disable_service_account_key_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Empty] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Empty]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def disable_service_account_key(name, disable_service_account_key_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:disable', options)
          command.request_representation = Google::Apis::IamV1::DisableServiceAccountKeyRequest::Representation
          command.request_object = disable_service_account_key_request_object
          command.response_representation = Google::Apis::IamV1::Empty::Representation
          command.response_class = Google::Apis::IamV1::Empty
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Enable a ServiceAccountKey.
        # @param [String] name
        #   Required. The resource name of the service account key. Use one of the
        #   following formats: * `projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`/
        #   keys/`KEY_ID`` * `projects/`PROJECT_ID`/serviceAccounts/`UNIQUE_ID`/keys/`
        #   KEY_ID`` As an alternative, you can use the `-` wildcard character instead of
        #   the project ID: * `projects/-/serviceAccounts/`EMAIL_ADDRESS`/keys/`KEY_ID`` *
        #   `projects/-/serviceAccounts/`UNIQUE_ID`/keys/`KEY_ID`` When possible, avoid
        #   using the `-` wildcard character, because it can cause response messages to
        #   contain misleading error codes. For example, if you try to access the service
        #   account key `projects/-/serviceAccounts/fake@example.com/keys/fake-key`, which
        #   does not exist, the response contains an HTTP `403 Forbidden` error instead of
        #   a `404 Not Found` error.
        # @param [Google::Apis::IamV1::EnableServiceAccountKeyRequest] enable_service_account_key_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Empty] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Empty]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def enable_service_account_key(name, enable_service_account_key_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}:enable', options)
          command.request_representation = Google::Apis::IamV1::EnableServiceAccountKeyRequest::Representation
          command.request_object = enable_service_account_key_request_object
          command.response_representation = Google::Apis::IamV1::Empty::Representation
          command.response_class = Google::Apis::IamV1::Empty
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets a ServiceAccountKey.
        # @param [String] name
        #   Required. The resource name of the service account key. Use one of the
        #   following formats: * `projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`/
        #   keys/`KEY_ID`` * `projects/`PROJECT_ID`/serviceAccounts/`UNIQUE_ID`/keys/`
        #   KEY_ID`` As an alternative, you can use the `-` wildcard character instead of
        #   the project ID: * `projects/-/serviceAccounts/`EMAIL_ADDRESS`/keys/`KEY_ID`` *
        #   `projects/-/serviceAccounts/`UNIQUE_ID`/keys/`KEY_ID`` When possible, avoid
        #   using the `-` wildcard character, because it can cause response messages to
        #   contain misleading error codes. For example, if you try to access the service
        #   account key `projects/-/serviceAccounts/fake@example.com/keys/fake-key`, which
        #   does not exist, the response contains an HTTP `403 Forbidden` error instead of
        #   a `404 Not Found` error.
        # @param [String] public_key_type
        #   Optional. The output format of the public key. The default is `TYPE_NONE`,
        #   which means that the public key is not returned.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ServiceAccountKey] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ServiceAccountKey]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_service_account_key(name, public_key_type: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::ServiceAccountKey::Representation
          command.response_class = Google::Apis::IamV1::ServiceAccountKey
          command.params['name'] = name unless name.nil?
          command.query['publicKeyType'] = public_key_type unless public_key_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists every ServiceAccountKey for a service account.
        # @param [String] name
        #   Required. The resource name of the service account. Use one of the following
        #   formats: * `projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/
        #   `PROJECT_ID`/serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-
        #   ` wildcard character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [Array<String>, String] key_types
        #   Filters the types of keys the user wants to include in the list response.
        #   Duplicate key types are not allowed. If no key type is provided, all keys are
        #   returned.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListServiceAccountKeysResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListServiceAccountKeysResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_service_account_keys(name, key_types: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}/keys', options)
          command.response_representation = Google::Apis::IamV1::ListServiceAccountKeysResponse::Representation
          command.response_class = Google::Apis::IamV1::ListServiceAccountKeysResponse
          command.params['name'] = name unless name.nil?
          command.query['keyTypes'] = key_types unless key_types.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Uploads the public key portion of a key pair that you manage, and associates
        # the public key with a ServiceAccount. After you upload the public key, you can
        # use the private key from the key pair as a service account key.
        # @param [String] name
        #   The resource name of the service account key. Use one of the following formats:
        #   * `projects/`PROJECT_ID`/serviceAccounts/`EMAIL_ADDRESS`` * `projects/`
        #   PROJECT_ID`/serviceAccounts/`UNIQUE_ID`` As an alternative, you can use the `-`
        #   wildcard character instead of the project ID: * `projects/-/serviceAccounts/`
        #   EMAIL_ADDRESS`` * `projects/-/serviceAccounts/`UNIQUE_ID`` When possible,
        #   avoid using the `-` wildcard character, because it can cause response messages
        #   to contain misleading error codes. For example, if you try to access the
        #   service account `projects/-/serviceAccounts/fake@example.com`, which does not
        #   exist, the response contains an HTTP `403 Forbidden` error instead of a `404
        #   Not Found` error.
        # @param [Google::Apis::IamV1::UploadServiceAccountKeyRequest] upload_service_account_key_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ServiceAccountKey] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ServiceAccountKey]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def upload_service_account_key(name, upload_service_account_key_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/{+name}/keys:upload', options)
          command.request_representation = Google::Apis::IamV1::UploadServiceAccountKeyRequest::Representation
          command.request_object = upload_service_account_key_request_object
          command.response_representation = Google::Apis::IamV1::ServiceAccountKey::Representation
          command.response_class = Google::Apis::IamV1::ServiceAccountKey
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the definition of a Role.
        # @param [String] name
        #   The `name` parameter's value depends on the target resource for the request,
        #   namely [roles](https://cloud.google.com/iam/docs/reference/rest/v1/roles), [
        #   projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.roles),
        #   or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `name` value format is described
        #   below: * [roles.get](https://cloud.google.com/iam/docs/reference/rest/v1/roles/
        #   get): `roles/`ROLE_NAME``. This method returns results from all [predefined
        #   roles](https://cloud.google.com/iam/docs/understanding-roles#predefined_roles)
        #   in IAM. Example request URL: `https://iam.googleapis.com/v1/roles/`ROLE_NAME``
        #   * [projects.roles.get](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   projects.roles/get): `projects/`PROJECT_ID`/roles/`CUSTOM_ROLE_ID``. This
        #   method returns only [custom roles](https://cloud.google.com/iam/docs/
        #   understanding-custom-roles) that have been created at the project level.
        #   Example request URL: `https://iam.googleapis.com/v1/projects/`PROJECT_ID`/
        #   roles/`CUSTOM_ROLE_ID`` * [organizations.roles.get](https://cloud.google.com/
        #   iam/docs/reference/rest/v1/organizations.roles/get): `organizations/`
        #   ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID``. This method returns only [custom
        #   roles](https://cloud.google.com/iam/docs/understanding-custom-roles) that have
        #   been created at the organization level. Example request URL: `https://iam.
        #   googleapis.com/v1/organizations/`ORGANIZATION_ID`/roles/`CUSTOM_ROLE_ID`` Note:
        #   Wildcard (*) values are invalid; you must specify a complete project ID or
        #   organization ID.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::Role] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::Role]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_role(name, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/{+name}', options)
          command.response_representation = Google::Apis::IamV1::Role::Representation
          command.response_class = Google::Apis::IamV1::Role
          command.params['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists every predefined Role that IAM supports, or every custom role that is
        # defined for an organization or project.
        # @param [Fixnum] page_size
        #   Optional limit on the number of roles to include in the response. The default
        #   is 300, and the maximum is 1,000.
        # @param [String] page_token
        #   Optional pagination token returned in an earlier ListRolesResponse.
        # @param [String] parent
        #   The `parent` parameter's value depends on the target resource for the request,
        #   namely [roles](https://cloud.google.com/iam/docs/reference/rest/v1/roles), [
        #   projects](https://cloud.google.com/iam/docs/reference/rest/v1/projects.roles),
        #   or [organizations](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   organizations.roles). Each resource type's `parent` value format is described
        #   below: * [roles.list](https://cloud.google.com/iam/docs/reference/rest/v1/
        #   roles/list): An empty string. This method doesn't require a resource; it
        #   simply returns all [predefined roles](https://cloud.google.com/iam/docs/
        #   understanding-roles#predefined_roles) in IAM. Example request URL: `https://
        #   iam.googleapis.com/v1/roles` * [projects.roles.list](https://cloud.google.com/
        #   iam/docs/reference/rest/v1/projects.roles/list): `projects/`PROJECT_ID``. This
        #   method lists all project-level [custom roles](https://cloud.google.com/iam/
        #   docs/understanding-custom-roles). Example request URL: `https://iam.googleapis.
        #   com/v1/projects/`PROJECT_ID`/roles` * [organizations.roles.list](https://cloud.
        #   google.com/iam/docs/reference/rest/v1/organizations.roles/list): `
        #   organizations/`ORGANIZATION_ID``. This method lists all organization-level [
        #   custom roles](https://cloud.google.com/iam/docs/understanding-custom-roles).
        #   Example request URL: `https://iam.googleapis.com/v1/organizations/`
        #   ORGANIZATION_ID`/roles` Note: Wildcard (*) values are invalid; you must
        #   specify a complete project ID or organization ID.
        # @param [Boolean] show_deleted
        #   Include Roles that have been deleted.
        # @param [String] view
        #   Optional view for the returned Role objects. When `FULL` is specified, the `
        #   includedPermissions` field is returned, which includes a list of all
        #   permissions in the role. The default value is `BASIC`, which does not return
        #   the `includedPermissions` field.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::ListRolesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::ListRolesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_roles(page_size: nil, page_token: nil, parent: nil, show_deleted: nil, view: nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:get, 'v1/roles', options)
          command.response_representation = Google::Apis::IamV1::ListRolesResponse::Representation
          command.response_class = Google::Apis::IamV1::ListRolesResponse
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['parent'] = parent unless parent.nil?
          command.query['showDeleted'] = show_deleted unless show_deleted.nil?
          command.query['view'] = view unless view.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists roles that can be granted on a Google Cloud resource. A role is
        # grantable if the IAM policy for the resource can contain bindings to the role.
        # @param [Google::Apis::IamV1::QueryGrantableRolesRequest] query_grantable_roles_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::IamV1::QueryGrantableRolesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::IamV1::QueryGrantableRolesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def query_grantable_roles(query_grantable_roles_request_object = nil, fields: nil, quota_user: nil, options: nil, &block)
          command = make_simple_command(:post, 'v1/roles:queryGrantableRoles', options)
          command.request_representation = Google::Apis::IamV1::QueryGrantableRolesRequest::Representation
          command.request_object = query_grantable_roles_request_object
          command.response_representation = Google::Apis::IamV1::QueryGrantableRolesResponse::Representation
          command.response_class = Google::Apis::IamV1::QueryGrantableRolesResponse
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          execute_or_queue_command(command, &block)
        end

        protected

        def apply_command_defaults(command)
          command.query['key'] = key unless key.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
        end
      end
    end
  end
end
