# frozen_string_literal: true

module KDL
  module Types
    class CountrySubdivision < Value::Custom
      # From: https://en.wikipedia.org/wiki/ISO_3166-2#Current_codes
      COUNTRY_SUBDIVISIONS = {
        "AD" => {
          "AD-07" => "Andorra la Vella",
          "AD-02" => "Canillo",
          "AD-03" => "Encamp",
          "AD-08" => "Escaldes-Engordany",
          "AD-04" => "La Massana",
          "AD-05" => "Ordino",
          "AD-06" => "Sant Julià de Lòria"
        },
        "AE" => {
          "AE-AZ" => "Abū Z̧aby",
          "AE-AJ" => "‘Ajmān",
          "AE-FU" => "Al Fujayrah",
          "AE-SH" => "Ash Shāriqah",
          "AE-DU" => "Dubayy",
          "AE-RK" => "Ra’s al Khaymah",
          "AE-UQ" => "Umm al Qaywayn"
        },
        "AF" => {
          "AF-BDS" => "Badakhshān",
          "AF-BDG" => "Bādghīs",
          "AF-BGL" => "Baghlān",
          "AF-BAL" => "Balkh",
          "AF-BAM" => "Bāmyān",
          "AF-DAY" => "Dāykundī",
          "AF-FRA" => "Farāh",
          "AF-FYB" => "Fāryāb",
          "AF-GHA" => "Ghaznī",
          "AF-GHO" => "Ghōr",
          "AF-HEL" => "Helmand",
          "AF-HER" => "Herāt",
          "AF-JOW" => "Jowzjān",
          "AF-KAB" => "Kābul",
          "AF-KAN" => "Kandahār",
          "AF-KAP" => "Kāpīsā",
          "AF-KHO" => "Khōst",
          "AF-KNR" => "Kunaṟ",
          "AF-KDZ" => "Kunduz",
          "AF-LAG" => "Laghmān",
          "AF-LOG" => "Lōgar",
          "AF-NAN" => "Nangarhār",
          "AF-NIM" => "Nīmrōz",
          "AF-NUR" => "Nūristān",
          "AF-PKA" => "Paktīkā",
          "AF-PIA" => "Paktiyā",
          "AF-PAN" => "Panjshayr",
          "AF-PAR" => "Parwān",
          "AF-SAM" => "Samangān",
          "AF-SAR" => "Sar-e Pul",
          "AF-TAK" => "Takhār",
          "AF-URU" => "Uruzgān",
          "AF-WAR" => "Wardak",
          "AF-ZAB" => "Zābul"
        },
        "AG" => {
          "AG-03" => "Saint George",
          "AG-04" => "Saint John",
          "AG-05" => "Saint Mary",
          "AG-06" => "Saint Paul",
          "AG-07" => "Saint Peter",
          "AG-08" => "Saint Philip",
          "AG-10" => "Barbuda",
          "AG-11" => "Redonda"
        },
        "AL" => {
          "AL-01" => "Berat",
          "AL-09" => "Dibër",
          "AL-02" => "Durrës",
          "AL-03" => "Elbasan",
          "AL-04" => "Fier",
          "AL-05" => "Gjirokastër",
          "AL-06" => "Korçë",
          "AL-07" => "Kukës",
          "AL-08" => "Lezhë",
          "AL-10" => "Shkodër",
          "AL-11" => "Tiranë",
          "AL-12" => "Vlorë"
        },
        "AM" => {
          "AM-ER" => "Erevan",
          "AM-AG" => "Aragac̣otn",
          "AM-AR" => "Ararat",
          "AM-AV" => "Armavir",
          "AM-GR" => "Geġark'unik'",
          "AM-KT" => "Kotayk'",
          "AM-LO" => "Loṙi",
          "AM-SH" => "Širak",
          "AM-SU" => "Syunik'",
          "AM-TV" => "Tavuš",
          "AM-VD" => "Vayoć Jor"
        },
        "AO" => {
          "AO-BGO" => "Bengo",
          "AO-BGU" => "Benguela",
          "AO-BIE" => "Bié",
          "AO-CAB" => "Cabinda",
          "AO-CNN" => "Cunene",
          "AO-HUA" => "Huambo",
          "AO-HUI" => "Huíla",
          "AO-CCU" => "Cuando Cubango",
          "AO-CNO" => "Cuanza-Norte",
          "AO-CUS" => "Cuanza-Sul",
          "AO-LUA" => "Luanda",
          "AO-LNO" => "Lunda-Norte",
          "AO-LSU" => "Lunda-Sul",
          "AO-MAL" => "Malange",
          "AO-MOX" => "Moxico",
          "AO-NAM" => "Namibe",
          "AO-UIG" => "Uíge",
          "AO-ZAI" => "Zaire"
        },
        "AR" => {
          "AR-C" => "Ciudad Autónoma de Buenos Aires",
          "AR-B" => "Buenos Aires",
          "AR-K" => "Catamarca",
          "AR-H" => "Chaco",
          "AR-U" => "Chubut",
          "AR-X" => "Córdoba",
          "AR-W" => "Corrientes",
          "AR-E" => "Entre Ríos",
          "AR-P" => "Formosa",
          "AR-Y" => "Jujuy",
          "AR-L" => "La Pampa",
          "AR-F" => "La Rioja",
          "AR-M" => "Mendoza",
          "AR-N" => "Misiones",
          "AR-Q" => "Neuquén",
          "AR-R" => "Río Negro",
          "AR-A" => "Salta",
          "AR-J" => "San Juan",
          "AR-D" => "San Luis",
          "AR-Z" => "Santa Cruz",
          "AR-S" => "Santa Fe",
          "AR-G" => "Santiago del Estero",
          "AR-V" => "Tierra del Fuego",
          "AR-T" => "Tucumán"
        },
        "AT" => {
          "AT-1" => "Burgenland",
          "AT-2" => "Kärnten",
          "AT-3" => "Niederösterreich",
          "AT-4" => "Oberösterreich",
          "AT-5" => "Salzburg",
          "AT-6" => "Steiermark",
          "AT-7" => "Tirol",
          "AT-8" => "Vorarlberg",
          "AT-9" => "Wien"
        },
        "AU" => {
          "AU-NSW" => "New South Wales",
          "AU-QLD" => "Queensland",
          "AU-SA" => "South Australia",
          "AU-TAS" => "Tasmania",
          "AU-VIC" => "Victoria",
          "AU-WA" => "Western Australia",
          "AU-ACT" => "Australian Capital Territory",
          "AU-NT" => "Northern Territory"
        },
        "AZ" => {
          "AZ-NX" => "Naxçıvan"
        },
        "BA" => {
          "BA-BIH" => "Federacija Bosne i Hercegovine",
          "BA-SRP" => "Republika Srpska",
          "BA-BRC" => "Brčko distrikt"
        },
        "BB" => {
          "BB-01" => "Christ Church",
          "BB-02" => "Saint Andrew",
          "BB-03" => "Saint George",
          "BB-04" => "Saint James",
          "BB-05" => "Saint John",
          "BB-06" => "Saint Joseph",
          "BB-07" => "Saint Lucy",
          "BB-08" => "Saint Michael",
          "BB-09" => "Saint Peter",
          "BB-10" => "Saint Philip",
          "BB-11" => "Saint Thomas"
        },
        "BD" => {
          "BD-A" => "Barishal",
          "BD-B" => "Chattogram",
          "BD-C" => "Dhaka",
          "BD-D" => "Khulna",
          "BD-H" => "Mymensingh",
          "BD-E" => "Rajshahi",
          "BD-F" => "Rangpur",
          "BD-G" => "Sylhet"
        },
        "BE" => {
          "BE-BRU" => "Brussels Hoofdstedelijk Gewest",
          "BE-VLG" => "Vlaams Gewest",
          "BE-WAL" => "Waals Gewest [note 3]"
        },
        "BF" => {
          "BF-01" => "Boucle du Mouhoun",
          "BF-02" => "Cascades",
          "BF-03" => "Centre",
          "BF-04" => "Centre-Est",
          "BF-05" => "Centre-Nord",
          "BF-06" => "Centre-Ouest",
          "BF-07" => "Centre-Sud",
          "BF-08" => "Est",
          "BF-09" => "Hauts-Bassins",
          "BF-10" => "Nord",
          "BF-11" => "Plateau-Central",
          "BF-12" => "Sahel",
          "BF-13" => "Sud-Ouest"
        },
        "BG" => {
          "BG-01" => "Blagoevgrad",
          "BG-02" => "Burgas",
          "BG-08" => "Dobrich",
          "BG-07" => "Gabrovo",
          "BG-26" => "Haskovo",
          "BG-09" => "Kardzhali",
          "BG-10" => "Kyustendil",
          "BG-11" => "Lovech",
          "BG-12" => "Montana",
          "BG-13" => "Pazardzhik",
          "BG-14" => "Pernik",
          "BG-15" => "Pleven",
          "BG-16" => "Plovdiv",
          "BG-17" => "Razgrad",
          "BG-18" => "Ruse",
          "BG-27" => "Shumen",
          "BG-19" => "Silistra",
          "BG-20" => "Sliven",
          "BG-21" => "Smolyan",
          "BG-23" => "Sofia",
          "BG-22" => "Sofia (stolitsa)",
          "BG-24" => "Stara Zagora",
          "BG-25" => "Targovishte",
          "BG-03" => "Varna",
          "BG-04" => "Veliko Tarnovo",
          "BG-05" => "Vidin",
          "BG-06" => "Vratsa",
          "BG-28" => "Yambol"
        },
        "BH" => {
          "BH-13" => "Al ‘Āşimah",
          "BH-14" => "Al Janūbīyah",
          "BH-15" => "Al Muḩarraq",
          "BH-17" => "Ash Shamālīyah"
        },
        "BI" => {
          "BI-BB" => "Bubanza",
          "BI-BM" => "Bujumbura Mairie",
          "BI-BL" => "Bujumbura Rural",
          "BI-BR" => "Bururi",
          "BI-CA" => "Cankuzo",
          "BI-CI" => "Cibitoke",
          "BI-GI" => "Gitega",
          "BI-KR" => "Karuzi",
          "BI-KY" => "Kayanza",
          "BI-KI" => "Kirundo",
          "BI-MA" => "Makamba",
          "BI-MU" => "Muramvya",
          "BI-MY" => "Muyinga",
          "BI-MW" => "Mwaro",
          "BI-NG" => "Ngozi",
          "BI-RM" => "Rumonge",
          "BI-RT" => "Rutana",
          "BI-RY" => "Ruyigi"
        },
        "BJ" => {
          "BJ-AL" => "Alibori",
          "BJ-AK" => "Atacora",
          "BJ-AQ" => "Atlantique",
          "BJ-BO" => "Borgou",
          "BJ-CO" => "Collines",
          "BJ-KO" => "Couffo",
          "BJ-DO" => "Donga",
          "BJ-LI" => "Littoral",
          "BJ-MO" => "Mono",
          "BJ-OU" => "Ouémé",
          "BJ-PL" => "Plateau",
          "BJ-ZO" => "Zou"
        },
        "BN" => {
          "BN-BE" => "Belait",
          "BN-BM" => "Brunei-Muara",
          "BN-TE" => "Temburong",
          "BN-TU" => "Tutong"
        },
        "BO" => {
          "BO-C" => "Cochabamba",
          "BO-H" => "Chuquisaca",
          "BO-B" => "El Beni",
          "BO-L" => "La Paz",
          "BO-O" => "Oruro",
          "BO-N" => "Pando",
          "BO-P" => "Potosí",
          "BO-S" => "Santa Cruz",
          "BO-T" => "Tarija"
        },
        "BQ" => {
          "BQ-BO" => "Bonaire",
          "BQ-SA" => "Saba",
          "BQ-SE" => "Sint Eustatius"
        },
        "BR" => {
          "BR-AC" => "Acre",
          "BR-AL" => "Alagoas",
          "BR-AP" => "Amapá",
          "BR-AM" => "Amazonas",
          "BR-BA" => "Bahia",
          "BR-CE" => "Ceará",
          "BR-DF" => "Distrito Federal",
          "BR-ES" => "Espírito Santo",
          "BR-GO" => "Goiás",
          "BR-MA" => "Maranhão",
          "BR-MT" => "Mato Grosso",
          "BR-MS" => "Mato Grosso do Sul",
          "BR-MG" => "Minas Gerais",
          "BR-PA" => "Pará",
          "BR-PB" => "Paraíba",
          "BR-PR" => "Paraná",
          "BR-PE" => "Pernambuco",
          "BR-PI" => "Piauí",
          "BR-RJ" => "Rio de Janeiro",
          "BR-RN" => "Rio Grande do Norte",
          "BR-RS" => "Rio Grande do Sul",
          "BR-RO" => "Rondônia",
          "BR-RR" => "Roraima",
          "BR-SC" => "Santa Catarina",
          "BR-SP" => "São Paulo",
          "BR-SE" => "Sergipe",
          "BR-TO" => "Tocantins"
        },
        "BS" => {
          "BS-AK" => "Acklins",
          "BS-BY" => "Berry Islands",
          "BS-BI" => "Bimini",
          "BS-BP" => "Black Point",
          "BS-CI" => "Cat Island",
          "BS-CO" => "Central Abaco",
          "BS-CS" => "Central Andros",
          "BS-CE" => "Central Eleuthera",
          "BS-FP" => "City of Freeport",
          "BS-CK" => "Crooked Island and Long Cay",
          "BS-EG" => "East Grand Bahama",
          "BS-EX" => "Exuma",
          "BS-GC" => "Grand Cay",
          "BS-HI" => "Harbour Island",
          "BS-HT" => "Hope Town",
          "BS-IN" => "Inagua",
          "BS-LI" => "Long Island",
          "BS-MC" => "Mangrove Cay",
          "BS-MG" => "Mayaguana",
          "BS-MI" => "Moore's Island",
          "BS-NP" => "New Providence",
          "BS-NO" => "North Abaco",
          "BS-NS" => "North Andros",
          "BS-NE" => "North Eleuthera",
          "BS-RI" => "Ragged Island",
          "BS-RC" => "Rum Cay",
          "BS-SS" => "San Salvador",
          "BS-SO" => "South Abaco",
          "BS-SA" => "South Andros",
          "BS-SE" => "South Eleuthera",
          "BS-SW" => "Spanish Wells",
          "BS-WG" => "West Grand Bahama"
        },
        "BT" => {
          "BT-33" => "Bumthang",
          "BT-12" => "Chhukha",
          "BT-22" => "Dagana",
          "BT-GA" => "Gasa",
          "BT-13" => "Haa",
          "BT-44" => "Lhuentse",
          "BT-42" => "Monggar",
          "BT-11" => "Paro",
          "BT-43" => "Pema Gatshel",
          "BT-23" => "Punakha",
          "BT-45" => "Samdrup Jongkhar",
          "BT-14" => "Samtse",
          "BT-31" => "Sarpang",
          "BT-15" => "Thimphu",
          "BT-41" => "Trashigang",
          "BT-TY" => "Trashi Yangtse",
          "BT-32" => "Trongsa",
          "BT-21" => "Tsirang",
          "BT-24" => "Wangdue Phodrang",
          "BT-34" => "Zhemgang"
        },
        "BW" => {
          "BW-CE" => "Central",
          "BW-CH" => "Chobe",
          "BW-FR" => "Francistown",
          "BW-GA" => "Gaborone",
          "BW-GH" => "Ghanzi",
          "BW-JW" => "Jwaneng",
          "BW-KG" => "Kgalagadi",
          "BW-KL" => "Kgatleng",
          "BW-KW" => "Kweneng",
          "BW-LO" => "Lobatse",
          "BW-NE" => "North East",
          "BW-NW" => "North West",
          "BW-SP" => "Selibe Phikwe",
          "BW-SE" => "South East",
          "BW-SO" => "Southern",
          "BW-ST" => "Sowa Town"
        },
        "BY" => {
          "BY-BR" => "Brestskaya voblasts'",
          "BY-HO" => "Homyel'skaya voblasts'",
          "BY-HR" => "Hrodzyenskaya voblasts'",
          "BY-MA" => "Mahilyowskaya voblasts'",
          "BY-MI" => "Minskaya voblasts'",
          "BY-VI" => "Vitsyebskaya voblasts'",
          "BY-HM" => "Horad Minsk"
        },
        "BZ" => {
          "BZ-BZ" => "Belize",
          "BZ-CY" => "Cayo",
          "BZ-CZL" => "Corozal",
          "BZ-OW" => "Orange Walk",
          "BZ-SC" => "Stann Creek",
          "BZ-TOL" => "Toledo"
        },
        "CA" => {
          "CA-AB" => "Alberta",
          "CA-BC" => "British Columbia",
          "CA-MB" => "Manitoba",
          "CA-NB" => "New Brunswick",
          "CA-NL" => "Newfoundland and Labrador",
          "CA-NS" => "Nova Scotia",
          "CA-ON" => "Ontario",
          "CA-PE" => "Prince Edward Island",
          "CA-QC" => "Quebec",
          "CA-SK" => "Saskatchewan",
          "CA-NT" => "Northwest Territories",
          "CA-NU" => "Nunavut",
          "CA-YT" => "Yukon"
        },
        "CD" => {
          "CD-KN" => "Kinshasa",
          "CD-BC" => "Kongo Central",
          "CD-EQ" => "Équateur",
          "CD-KE" => "Kasaï Oriental",
          "CD-MA" => "Maniema",
          "CD-NK" => "Nord-Kivu",
          "CD-SK" => "Sud-Kivu",
          "CD-BU" => "Bas-Uélé",
          "CD-HK" => "Haut-Katanga",
          "CD-HL" => "Haut-Lomami",
          "CD-HU" => "Haut-Uélé",
          "CD-IT" => "Ituri",
          "CD-KC" => "Kasaï Central",
          "CD-KG" => "Kwango",
          "CD-KL" => "Kwilu",
          "CD-KS" => "Kasaï",
          "CD-LO" => "Lomami",
          "CD-LU" => "Lualaba",
          "CD-MN" => "Mai-Ndombe",
          "CD-MO" => "Mongala",
          "CD-NU" => "Nord-Ubangi",
          "CD-SA" => "Sankuru",
          "CD-SU" => "Sud-Ubangi",
          "CD-TA" => "Tanganyika",
          "CD-TO" => "Tshopo",
          "CD-TU" => "Tshuapa"
        },
        "CF" => {
          "CF-BGF" => "Bangui",
          "CF-BB" => "Bamingui-Bangoran",
          "CF-BK" => "Basse-Kotto",
          "CF-HK" => "Haute-Kotto",
          "CF-HM" => "Haut-Mbomou",
          "CF-HS" => "Haute-Sangha/Mambéré-Kadéï",
          "CF-KG" => "Kémo-Gribingui",
          "CF-LB" => "Lobaye",
          "CF-MB" => "Mbomou",
          "CF-NM" => "Nana-Mambéré",
          "CF-MP" => "Ombella-Mpoko",
          "CF-UK" => "Ouaka",
          "CF-AC" => "Ouham",
          "CF-OP" => "Ouham-Pendé",
          "CF-VK" => "Vakaga",
          "CF-KB" => "Gribingui",
          "CF-SE" => "Sangha"
        },
        "CG" => {
          "CG-BZV" => "Brazzaville",
          "CG-11" => "Bouenza",
          "CG-8" => "Cuvette",
          "CG-15" => "Cuvette-Ouest",
          "CG-5" => "Kouilou",
          "CG-2" => "Lékoumou",
          "CG-7" => "Likouala",
          "CG-9" => "Niari",
          "CG-14" => "Plateaux",
          "CG-16" => "Pointe-Noire",
          "CG-12" => "Pool",
          "CG-13" => "Sangha"
        },
        "CH" => {
          "CH-AG" => "Aargau (de)",
          "CH-AR" => "Appenzell Ausserrhoden (de)",
          "CH-AI" => "Appenzell Innerrhoden (de)",
          "CH-BL" => "Basel-Landschaft (de)",
          "CH-BS" => "Basel-Stadt (de)",
          "CH-BE" => "Bern (de), Berne (fr)",
          "CH-FR" => "Fribourg (fr), Freiburg (de)",
          "CH-GE" => "Genève (fr)",
          "CH-GL" => "Glarus (de)",
          "CH-GR" => "Graubünden (de), Grigioni (it), Grischun (rm)",
          "CH-JU" => "Jura (fr)",
          "CH-LU" => "Luzern (de)",
          "CH-NE" => "Neuchâtel (fr)",
          "CH-NW" => "Nidwalden (de)",
          "CH-OW" => "Obwalden (de)",
          "CH-SG" => "Sankt Gallen (de)",
          "CH-SH" => "Schaffhausen (de)",
          "CH-SZ" => "Schwyz (de)",
          "CH-SO" => "Solothurn (de)",
          "CH-TG" => "Thurgau (de)",
          "CH-TI" => "Ticino (it)",
          "CH-UR" => "Uri (de)",
          "CH-VS" => "Valais (fr), Wallis (de)",
          "CH-VD" => "Vaud (fr)",
          "CH-ZG" => "Zug (de)",
          "CH-ZH" => "Zürich (de)"
        },
        "CI" => {
          "CI-AB" => "Abidjan",
          "CI-BS" => "Bas-Sassandra",
          "CI-CM" => "Comoé",
          "CI-DN" => "Denguélé",
          "CI-GD" => "Gôh-Djiboua",
          "CI-LC" => "Lacs",
          "CI-LG" => "Lagunes",
          "CI-MG" => "Montagnes",
          "CI-SM" => "Sassandra-Marahoué",
          "CI-SV" => "Savanes",
          "CI-VB" => "Vallée du Bandama",
          "CI-WR" => "Woroba",
          "CI-YM" => "Yamoussoukro",
          "CI-ZZ" => "Zanzan"
        },
        "CL" => {
          "CL-AI" => "Aisén del General Carlos Ibañez del Campo",
          "CL-AN" => "Antofagasta",
          "CL-AP" => "Arica y Parinacota",
          "CL-AR" => "La Araucanía",
          "CL-AT" => "Atacama",
          "CL-BI" => "Biobío",
          "CL-CO" => "Coquimbo",
          "CL-LI" => "Libertador General Bernardo O'Higgins",
          "CL-LL" => "Los Lagos",
          "CL-LR" => "Los Ríos",
          "CL-MA" => "Magallanes",
          "CL-ML" => "Maule",
          "CL-NB" => "Ñuble",
          "CL-RM" => "Región Metropolitana de Santiago",
          "CL-TA" => "Tarapacá",
          "CL-VS" => "Valparaíso"
        },
        "CM" => {
          "CM-AD" => "Adamaoua",
          "CM-CE" => "Centre",
          "CM-ES" => "East",
          "CM-EN" => "Far North",
          "CM-LT" => "Littoral",
          "CM-NO" => "North",
          "CM-NW" => "North-West",
          "CM-SU" => "South",
          "CM-SW" => "South-West",
          "CM-OU" => "West"
        },
        "CN" => {
          "CN-AH" => "Anhui Sheng",
          "CN-BJ" => "Beijing Shi",
          "CN-CQ" => "Chongqing Shi",
          "CN-FJ" => "Fujian Sheng",
          "CN-GD" => "Guangdong Sheng",
          "CN-GS" => "Gansu Sheng",
          "CN-GX" => "Guangxi Zhuangzu Zizhiqu",
          "CN-GZ" => "Guizhou Sheng",
          "CN-HA" => "Henan Sheng",
          "CN-HB" => "Hubei Sheng",
          "CN-HE" => "Hebei Sheng",
          "CN-HI" => "Hainan Sheng",
          "CN-HK" => "Hong Kong SAR (en)Xianggang Tebiexingzhengqu (zh)",
          "CN-HL" => "Heilongjiang Sheng",
          "CN-HN" => "Hunan Sheng",
          "CN-JL" => "Jilin Sheng",
          "CN-JS" => "Jiangsu Sheng",
          "CN-JX" => "Jiangxi Sheng",
          "CN-LN" => "Liaoning Sheng",
          "CN-MO" => "Macao SAR (en)Macau SAR (pt)Aomen Tebiexingzhengqu (zh)",
          "CN-NM" => "Nei Mongol Zizhiqu",
          "CN-NX" => "Ningxia Huizi Zizhiqu",
          "CN-QH" => "Qinghai Sheng",
          "CN-SC" => "Sichuan Sheng",
          "CN-SD" => "Shandong Sheng",
          "CN-SH" => "Shanghai Shi",
          "CN-SN" => "Shaanxi Sheng",
          "CN-SX" => "Shanxi Sheng",
          "CN-TJ" => "Tianjin Shi",
          "CN-TW" => "Taiwan Sheng",
          "CN-XJ" => "Xinjiang Uygur Zizhiqu",
          "CN-XZ" => "Xizang Zizhiqu",
          "CN-YN" => "Yunnan Sheng",
          "CN-ZJ" => "Zhejiang Sheng"
        },
        "CO" => {
          "CO-DC" => "Distrito Capital de Bogotá",
          "CO-AMA" => "Amazonas",
          "CO-ANT" => "Antioquia",
          "CO-ARA" => "Arauca",
          "CO-ATL" => "Atlántico",
          "CO-BOL" => "Bolívar",
          "CO-BOY" => "Boyacá",
          "CO-CAL" => "Caldas",
          "CO-CAQ" => "Caquetá",
          "CO-CAS" => "Casanare",
          "CO-CAU" => "Cauca",
          "CO-CES" => "Cesar",
          "CO-COR" => "Córdoba",
          "CO-CUN" => "Cundinamarca",
          "CO-CHO" => "Chocó",
          "CO-GUA" => "Guainía",
          "CO-GUV" => "Guaviare",
          "CO-HUI" => "Huila",
          "CO-LAG" => "La Guajira",
          "CO-MAG" => "Magdalena",
          "CO-MET" => "Meta",
          "CO-NAR" => "Nariño",
          "CO-NSA" => "Norte de Santander",
          "CO-PUT" => "Putumayo",
          "CO-QUI" => "Quindío",
          "CO-RIS" => "Risaralda",
          "CO-SAP" => "San Andrés, Providencia y Santa Catalina",
          "CO-SAN" => "Santander",
          "CO-SUC" => "Sucre",
          "CO-TOL" => "Tolima",
          "CO-VAC" => "Valle del Cauca",
          "CO-VAU" => "Vaupés",
          "CO-VID" => "Vichada"
        },
        "CR" => {
          "CR-A" => "Alajuela",
          "CR-C" => "Cartago",
          "CR-G" => "Guanacaste",
          "CR-H" => "Heredia",
          "CR-L" => "Limón",
          "CR-P" => "Puntarenas",
          "CR-SJ" => "San José"
        },
        "CU" => {
          "CU-15" => "Artemisa",
          "CU-09" => "Camagüey",
          "CU-08" => "Ciego de Ávila",
          "CU-06" => "Cienfuegos",
          "CU-12" => "Granma",
          "CU-14" => "Guantánamo",
          "CU-11" => "Holguín",
          "CU-03" => "La Habana",
          "CU-10" => "Las Tunas",
          "CU-04" => "Matanzas",
          "CU-16" => "Mayabeque",
          "CU-01" => "Pinar del Río",
          "CU-07" => "Sancti Spíritus",
          "CU-13" => "Santiago de Cuba",
          "CU-05" => "Villa Clara",
          "CU-99" => "Isla de la Juventud"
        },
        "CV" => {
          "CV-B" => "Ilhas de Barlavento",
          "CV-S" => "Ilhas de Sotavento"
        },
        "CY" => {
          "CY-04" => "Ammochostos",
          "CY-06" => "Keryneia",
          "CY-03" => "Larnaka",
          "CY-01" => "Lefkosia",
          "CY-02" => "Lemesos",
          "CY-05" => "Pafos"
        },
        "CZ" => {
          "CZ-31" => "Jihočeský kraj",
          "CZ-64" => "Jihomoravský kraj",
          "CZ-41" => "Karlovarský kraj",
          "CZ-52" => "Královéhradecký kraj",
          "CZ-51" => "Liberecký kraj",
          "CZ-80" => "Moravskoslezský kraj",
          "CZ-71" => "Olomoucký kraj",
          "CZ-53" => "Pardubický kraj",
          "CZ-32" => "Plzeňský kraj",
          "CZ-10" => "Praha, Hlavní město",
          "CZ-20" => "Středočeský kraj",
          "CZ-42" => "Ústecký kraj",
          "CZ-63" => "Kraj Vysočina",
          "CZ-72" => "Zlínský kraj"
        },
        "DE" => {
          "DE-BW" => "Baden-Württemberg",
          "DE-BY" => "Bayern",
          "DE-BE" => "Berlin",
          "DE-BB" => "Brandenburg",
          "DE-HB" => "Bremen",
          "DE-HH" => "Hamburg",
          "DE-HE" => "Hessen",
          "DE-MV" => "Mecklenburg-Vorpommern",
          "DE-NI" => "Niedersachsen",
          "DE-NW" => "Nordrhein-Westfalen",
          "DE-RP" => "Rheinland-Pfalz",
          "DE-SL" => "Saarland",
          "DE-SN" => "Sachsen",
          "DE-ST" => "Sachsen-Anhalt",
          "DE-SH" => "Schleswig-Holstein",
          "DE-TH" => "Thüringen"
        },
        "DJ" => {
          "DJ-AS" => "Ali Sabieh",
          "DJ-AR" => "Arta",
          "DJ-DI" => "Dikhil",
          "DJ-OB" => "Obock",
          "DJ-TA" => "Tadjourah",
          "DJ-DJ" => "Djibouti"
        },
        "DK" => {
          "DK-84" => "Hovedstaden",
          "DK-82" => "Midtjylland",
          "DK-81" => "Nordjylland",
          "DK-85" => "Sjælland",
          "DK-83" => "Syddanmark"
        },
        "DM" => {
          "DM-02" => "Saint Andrew",
          "DM-03" => "Saint David",
          "DM-04" => "Saint George",
          "DM-05" => "Saint John",
          "DM-06" => "Saint Joseph",
          "DM-07" => "Saint Luke",
          "DM-08" => "Saint Mark",
          "DM-09" => "Saint Patrick",
          "DM-10" => "Saint Paul",
          "DM-11" => "Saint Peter"
        },
        "DO" => {
          "DO-33" => "Cibao Nordeste",
          "DO-34" => "Cibao Noroeste",
          "DO-35" => "Cibao Norte",
          "DO-36" => "Cibao Sur",
          "DO-37" => "El Valle",
          "DO-38" => "Enriquillo",
          "DO-39" => "Higuamo",
          "DO-40" => "Ozama",
          "DO-41" => "Valdesia",
          "DO-42" => "Yuma"
        },
        "DZ" => {
          "DZ-01" => "Adrar",
          "DZ-44" => "Aïn Defla",
          "DZ-46" => "Aïn Témouchent",
          "DZ-16" => "Alger",
          "DZ-23" => "Annaba",
          "DZ-05" => "Batna",
          "DZ-08" => "Béchar",
          "DZ-06" => "Béjaïa",
          "DZ-07" => "Biskra",
          "DZ-09" => "Blida",
          "DZ-34" => "Bordj Bou Arréridj",
          "DZ-10" => "Bouira",
          "DZ-35" => "Boumerdès",
          "DZ-02" => "Chlef",
          "DZ-25" => "Constantine",
          "DZ-17" => "Djelfa",
          "DZ-32" => "El Bayadh",
          "DZ-39" => "El Oued",
          "DZ-36" => "El Tarf",
          "DZ-47" => "Ghardaïa",
          "DZ-24" => "Guelma",
          "DZ-33" => "Illizi",
          "DZ-18" => "Jijel",
          "DZ-40" => "Khenchela",
          "DZ-03" => "Laghouat",
          "DZ-28" => "M'sila",
          "DZ-29" => "Mascara",
          "DZ-26" => "Médéa",
          "DZ-43" => "Mila",
          "DZ-27" => "Mostaganem",
          "DZ-45" => "Naama",
          "DZ-31" => "Oran",
          "DZ-30" => "Ouargla",
          "DZ-04" => "Oum el Bouaghi",
          "DZ-48" => "Relizane",
          "DZ-20" => "Saïda",
          "DZ-19" => "Sétif",
          "DZ-22" => "Sidi Bel Abbès",
          "DZ-21" => "Skikda",
          "DZ-41" => "Souk Ahras",
          "DZ-11" => "Tamanrasset",
          "DZ-12" => "Tébessa",
          "DZ-14" => "Tiaret",
          "DZ-37" => "Tindouf",
          "DZ-42" => "Tipaza",
          "DZ-38" => "Tissemsilt",
          "DZ-15" => "Tizi Ouzou",
          "DZ-13" => "Tlemcen"
        },
        "EC" => {
          "EC-A" => "Azuay",
          "EC-B" => "Bolívar",
          "EC-F" => "Cañar",
          "EC-C" => "Carchi",
          "EC-H" => "Chimborazo",
          "EC-X" => "Cotopaxi",
          "EC-O" => "El Oro",
          "EC-E" => "Esmeraldas",
          "EC-W" => "Galápagos",
          "EC-G" => "Guayas",
          "EC-I" => "Imbabura",
          "EC-L" => "Loja",
          "EC-R" => "Los Ríos",
          "EC-M" => "Manabí",
          "EC-S" => "Morona Santiago",
          "EC-N" => "Napo",
          "EC-D" => "Orellana",
          "EC-Y" => "Pastaza",
          "EC-P" => "Pichincha",
          "EC-SE" => "Santa Elena",
          "EC-SD" => "Santo Domingo de los Tsáchilas",
          "EC-U" => "Sucumbíos",
          "EC-T" => "Tungurahua",
          "EC-Z" => "Zamora Chinchipe"
        },
        "EE" => {
          "EE-37" => "Harjumaa",
          "EE-39" => "Hiiumaa",
          "EE-45" => "Ida-Virumaa",
          "EE-50" => "Jõgevamaa",
          "EE-52" => "Järvamaa",
          "EE-56" => "Läänemaa",
          "EE-60" => "Lääne-Virumaa",
          "EE-64" => "Põlvamaa",
          "EE-68" => "Pärnumaa",
          "EE-71" => "Raplamaa",
          "EE-74" => "Saaremaa",
          "EE-79" => "Tartumaa",
          "EE-81" => "Valgamaa",
          "EE-84" => "Viljandimaa",
          "EE-87" => "Võrumaa"
        },
        "EG" => {
          "EG-DK" => "Ad Daqahlīyah",
          "EG-BA" => "Al Baḩr al Aḩmar",
          "EG-BH" => "Al Buḩayrah",
          "EG-FYM" => "Al Fayyūm",
          "EG-GH" => "Al Gharbīyah",
          "EG-ALX" => "Al Iskandarīyah",
          "EG-IS" => "Al Ismā'īlīyah",
          "EG-GZ" => "Al Jīzah",
          "EG-MNF" => "Al Minūfīyah",
          "EG-MN" => "Al Minyā",
          "EG-C" => "Al Qāhirah",
          "EG-KB" => "Al Qalyūbīyah",
          "EG-LX" => "Al Uqşur",
          "EG-WAD" => "Al Wādī al Jadīd",
          "EG-SUZ" => "As Suways",
          "EG-SHR" => "Ash Sharqīyah",
          "EG-ASN" => "Aswān",
          "EG-AST" => "Asyūţ",
          "EG-BNS" => "Banī Suwayf",
          "EG-PTS" => "Būr Sa‘īd",
          "EG-DT" => "Dumyāţ",
          "EG-JS" => "Janūb Sīnā'",
          "EG-KFS" => "Kafr ash Shaykh",
          "EG-MT" => "Maţrūḩ",
          "EG-KN" => "Qinā",
          "EG-SIN" => "Shamāl Sīnā'",
          "EG-SHG" => "Sūhāj"
        },
        "ER" => {
          "ER-MA" => "Al Awsaţ",
          "ER-DU" => "Al Janūbī",
          "ER-AN" => "Ansabā",
          "ER-DK" => "Janūbī al Baḩrī al Aḩmar",
          "ER-GB" => "Qāsh-Barkah",
          "ER-SK" => "Shimālī al Baḩrī al Aḩmar"
        },
        "ES" => {
          "ES-AN" => "Andalucía",
          "ES-AR" => "Aragón",
          "ES-AS" => "Asturias, Principado de",
          "ES-CN" => "Canarias",
          "ES-CB" => "Cantabria",
          "ES-CL" => "Castilla y León",
          "ES-CM" => "Castilla-La Mancha",
          "ES-CT" => "Catalunya (ca) [Cataluña]",
          "ES-CE" => "Ceuta",
          "ES-EX" => "Extremadura",
          "ES-GA" => "Galicia (gl) [Galicia]",
          "ES-IB" => "Illes Balears (ca) [Islas Baleares]",
          "ES-RI" => "La Rioja",
          "ES-MD" => "Madrid, Comunidad de",
          "ES-ML" => "Melilla",
          "ES-MC" => "Murcia, Región de",
          "ES-NC" => "Navarra, Comunidad Foral deNafarroako Foru Komunitatea (eu)",
          "ES-PV" => "País VascoEuskal Herria (eu)",
          "ES-VC" => "Valenciana, Comunitat (ca) [Valenciana, Comunidad]"
        },
        "ET" => {
          "ET-AA" => "Ādīs Ābeba",
          "ET-DD" => "Dirē Dawa",
          "ET-AF" => "Āfar",
          "ET-AM" => "Āmara",
          "ET-BE" => "Bīnshangul Gumuz",
          "ET-GA" => "Gambēla Hizboch",
          "ET-HA" => "Hārerī Hizb",
          "ET-OR" => "Oromīya",
          "ET-SO" => "Sumalē",
          "ET-TI" => "Tigray",
          "ET-SN" => "YeDebub Bihēroch Bihēreseboch na Hizboch"
        },
        "FI" => {
          "FI-01" => "Ahvenanmaan maakunta",
          "FI-02" => "Etelä-Karjala",
          "FI-03" => "Etelä-Pohjanmaa",
          "FI-04" => "Etelä-Savo",
          "FI-05" => "Kainuu",
          "FI-06" => "Kanta-Häme",
          "FI-07" => "Keski-Pohjanmaa",
          "FI-08" => "Keski-Suomi",
          "FI-09" => "Kymenlaakso",
          "FI-10" => "Lappi",
          "FI-11" => "Pirkanmaa",
          "FI-12" => "Pohjanmaa",
          "FI-13" => "Pohjois-Karjala",
          "FI-14" => "Pohjois-Pohjanmaa",
          "FI-15" => "Pohjois-Savo",
          "FI-16" => "Päijät-Häme",
          "FI-17" => "Satakunta",
          "FI-18" => "Uusimaa",
          "FI-19" => "Varsinais-Suomi"
        },
        "FJ" => {
          "FJ-C" => "Central",
          "FJ-E" => "Eastern",
          "FJ-N" => "Northern",
          "FJ-W" => "Western",
          "FJ-R" => "Rotuma"
        },
        "FM" => {
          "FM-TRK" => "Chuuk",
          "FM-KSA" => "Kosrae",
          "FM-PNI" => "Pohnpei",
          "FM-YAP" => "Yap"
        },
        "FR" => {
          "FR-ARA" => "Auvergne-Rhône-Alpes",
          "FR-BFC" => "Bourgogne-Franche-Comté",
          "FR-BRE" => "Bretagne",
          "FR-CVL" => "Centre-Val de Loire",
          "FR-20R" => "Corse",
          "FR-GES" => "Grand-Est",
          "FR-GP" => "Guadeloupe",
          "FR-GF" => "Guyane (française)",
          "FR-HDF" => "Hauts-de-France",
          "FR-IDF" => "Île-de-France",
          "FR-MQ" => "Martinique",
          "FR-YT" => "Mayotte",
          "FR-NOR" => "Normandie",
          "FR-NAQ" => "Nouvelle-Aquitaine",
          "FR-OCC" => "Occitanie",
          "FR-PDL" => "Pays-de-la-Loire",
          "FR-PAC" => "Provence-Alpes-Côte-d’Azur",
          "FR-RE" => "La Réunion"
        },
        "GA" => {
          "GA-1" => "Estuaire",
          "GA-2" => "Haut-Ogooué",
          "GA-3" => "Moyen-Ogooué",
          "GA-4" => "Ngounié",
          "GA-5" => "Nyanga",
          "GA-6" => "Ogooué-Ivindo",
          "GA-7" => "Ogooué-Lolo",
          "GA-8" => "Ogooué-Maritime",
          "GA-9" => "Woleu-Ntem"
        },
        "GB" => {
          "GB-ENG" => "England",
          "GB-NIR" => "Northern Ireland",
          "GB-SCT" => "Scotland",
          "GB-WLS" => "Wales [Cymru GB-CYM]"
        },
        "GD" => {
          "GD-01" => "Saint Andrew",
          "GD-02" => "Saint David",
          "GD-03" => "Saint George",
          "GD-04" => "Saint John",
          "GD-05" => "Saint Mark",
          "GD-06" => "Saint Patrick",
          "GD-10" => "Southern Grenadine Islands"
        },
        "GE" => {
          "GE-AB" => "Abkhazia",
          "GE-AJ" => "Ajaria",
          "GE-TB" => "Tbilisi",
          "GE-GU" => "Guria",
          "GE-IM" => "Imereti",
          "GE-KA" => "K'akheti",
          "GE-KK" => "Kvemo Kartli",
          "GE-MM" => "Mtskheta-Mtianeti",
          "GE-RL" => "Rach'a-Lechkhumi-Kvemo Svaneti",
          "GE-SZ" => "Samegrelo-Zemo Svaneti",
          "GE-SJ" => "Samtskhe-Javakheti",
          "GE-SK" => "Shida Kartli"
        },
        "GH" => {
          "GH-AF" => "Ahafo",
          "GH-AH" => "Ashanti",
          "GH-BO" => "Bono",
          "GH-BE" => "Bono East",
          "GH-CP" => "Central",
          "GH-EP" => "Eastern",
          "GH-AA" => "Greater Accra",
          "GH-NE" => "North East",
          "GH-NP" => "Northern",
          "GH-OT" => "Oti",
          "GH-SV" => "Savannah",
          "GH-UE" => "Upper East",
          "GH-UW" => "Upper West",
          "GH-TV" => "Volta",
          "GH-WP" => "Western",
          "GH-WN" => "Western North"
        },
        "GL" => {
          "GL-AV" => "Avannaata Kommunia",
          "GL-KU" => "Kommune Kujalleq",
          "GL-QT" => "Kommune Qeqertalik",
          "GL-SM" => "Kommuneqarfik Sermersooq",
          "GL-QE" => "Qeqqata Kommunia"
        },
        "GM" => {
          "GM-B" => "Banjul",
          "GM-M" => "Central River",
          "GM-L" => "Lower River",
          "GM-N" => "North Bank",
          "GM-U" => "Upper River",
          "GM-W" => "Western"
        },
        "GN" => {
          "GN-B" => "Boké",
          "GN-F" => "Faranah",
          "GN-K" => "Kankan",
          "GN-D" => "Kindia",
          "GN-L" => "Labé",
          "GN-M" => "Mamou",
          "GN-N" => "Nzérékoré",
          "GN-C" => "Conakry"
        },
        "GQ" => {
          "GQ-C" => "Región Continental",
          "GQ-I" => "Región Insular"
        },
        "GR" => {
          "GR-A" => "Anatolikí Makedonía kai Thráki",
          "GR-I" => "Attikí",
          "GR-G" => "Dytikí Elláda",
          "GR-C" => "Dytikí Makedonía",
          "GR-F" => "Ionía Nísia",
          "GR-D" => "Ípeiros",
          "GR-B" => "Kentrikí Makedonía",
          "GR-M" => "Kríti",
          "GR-L" => "Nótio Aigaío",
          "GR-J" => "Pelopónnisos",
          "GR-H" => "Stereá Elláda",
          "GR-E" => "Thessalía",
          "GR-K" => "Vóreio Aigaío"
        },
        "GT" => {
          "GT-AV" => "Alta Verapaz",
          "GT-BV" => "Baja Verapaz",
          "GT-CM" => "Chimaltenango",
          "GT-CQ" => "Chiquimula",
          "GT-PR" => "El Progreso",
          "GT-ES" => "Escuintla",
          "GT-GU" => "Guatemala",
          "GT-HU" => "Huehuetenango",
          "GT-IZ" => "Izabal",
          "GT-JA" => "Jalapa",
          "GT-JU" => "Jutiapa",
          "GT-PE" => "Petén",
          "GT-QZ" => "Quetzaltenango",
          "GT-QC" => "Quiché",
          "GT-RE" => "Retalhuleu",
          "GT-SA" => "Sacatepéquez",
          "GT-SM" => "San Marcos",
          "GT-SR" => "Santa Rosa",
          "GT-SO" => "Sololá",
          "GT-SU" => "Suchitepéquez",
          "GT-TO" => "Totonicapán",
          "GT-ZA" => "Zacapa"
        },
        "GW" => {
          "GW-L" => "Leste",
          "GW-N" => "Norte",
          "GW-S" => "Sul"
        },
        "GY" => {
          "GY-BA" => "Barima-Waini",
          "GY-CU" => "Cuyuni-Mazaruni",
          "GY-DE" => "Demerara-Mahaica",
          "GY-EB" => "East Berbice-Corentyne",
          "GY-ES" => "Essequibo Islands-West Demerara",
          "GY-MA" => "Mahaica-Berbice",
          "GY-PM" => "Pomeroon-Supenaam",
          "GY-PT" => "Potaro-Siparuni",
          "GY-UD" => "Upper Demerara-Berbice",
          "GY-UT" => "Upper Takutu-Upper Essequibo"
        },
        "HN" => {
          "HN-AT" => "Atlántida",
          "HN-CH" => "Choluteca",
          "HN-CL" => "Colón",
          "HN-CM" => "Comayagua",
          "HN-CP" => "Copán",
          "HN-CR" => "Cortés",
          "HN-EP" => "El Paraíso",
          "HN-FM" => "Francisco Morazán",
          "HN-GD" => "Gracias a Dios",
          "HN-IN" => "Intibucá",
          "HN-IB" => "Islas de la Bahía",
          "HN-LP" => "La Paz",
          "HN-LE" => "Lempira",
          "HN-OC" => "Ocotepeque",
          "HN-OL" => "Olancho",
          "HN-SB" => "Santa Bárbara",
          "HN-VA" => "Valle",
          "HN-YO" => "Yoro"
        },
        "HR" => {
          "HR-21" => "Grad Zagreb",
          "HR-07" => "Bjelovarsko-bilogorska županija",
          "HR-12" => "Brodsko-posavska županija",
          "HR-19" => "Dubrovačko-neretvanska županija",
          "HR-18" => "Istarska županija",
          "HR-04" => "Karlovačka županija",
          "HR-06" => "Koprivničko-križevačka županija",
          "HR-02" => "Krapinsko-zagorska županija",
          "HR-09" => "Ličko-senjska županija",
          "HR-20" => "Međimurska županija",
          "HR-14" => "Osječko-baranjska županija",
          "HR-11" => "Požeško-slavonska županija",
          "HR-08" => "Primorsko-goranska županija",
          "HR-03" => "Sisačko-moslavačka županija",
          "HR-17" => "Splitsko-dalmatinska županija",
          "HR-15" => "Šibensko-kninska županija",
          "HR-05" => "Varaždinska županija",
          "HR-10" => "Virovitičko-podravska županija",
          "HR-16" => "Vukovarsko-srijemska županija",
          "HR-13" => "Zadarska županija",
          "HR-01" => "Zagrebačka županija"
        },
        "HT" => {
          "HT-AR" => "Artibonite",
          "HT-CE" => "Centre",
          "HT-GA" => "Grande’Anse",
          "HT-NI" => "Nippes",
          "HT-ND" => "Nord",
          "HT-NE" => "Nord-Est",
          "HT-NO" => "Nord-Ouest",
          "HT-OU" => "Ouest",
          "HT-SD" => "Sud",
          "HT-SE" => "Sud-Est"
        },
        "HU" => {
          "HU-BU" => "Budapest",
          "HU-BK" => "Bács-Kiskun",
          "HU-BA" => "Baranya",
          "HU-BE" => "Békés",
          "HU-BZ" => "Borsod-Abaúj-Zemplén",
          "HU-CS" => "Csongrád",
          "HU-FE" => "Fejér",
          "HU-GS" => "Győr-Moson-Sopron",
          "HU-HB" => "Hajdú-Bihar",
          "HU-HE" => "Heves",
          "HU-JN" => "Jász-Nagykun-Szolnok",
          "HU-KE" => "Komárom-Esztergom",
          "HU-NO" => "Nógrád",
          "HU-PE" => "Pest",
          "HU-SO" => "Somogy",
          "HU-SZ" => "Szabolcs-Szatmár-Bereg",
          "HU-TO" => "Tolna",
          "HU-VA" => "Vas",
          "HU-VE" => "Veszprém",
          "HU-ZA" => "Zala",
          "HU-BC" => "Békéscsaba",
          "HU-DE" => "Debrecen",
          "HU-DU" => "Dunaújváros",
          "HU-EG" => "Eger",
          "HU-ER" => "Érd",
          "HU-GY" => "Győr",
          "HU-HV" => "Hódmezővásárhely",
          "HU-KV" => "Kaposvár",
          "HU-KM" => "Kecskemét",
          "HU-MI" => "Miskolc",
          "HU-NK" => "Nagykanizsa",
          "HU-NY" => "Nyíregyháza",
          "HU-PS" => "Pécs",
          "HU-ST" => "Salgótarján",
          "HU-SN" => "Sopron",
          "HU-SD" => "Szeged",
          "HU-SF" => "Székesfehérvár",
          "HU-SS" => "Szekszárd",
          "HU-SK" => "Szolnok",
          "HU-SH" => "Szombathely",
          "HU-TB" => "Tatabánya",
          "HU-VM" => "Veszprém",
          "HU-ZE" => "Zalaegerszeg"
        },
        "ID" => {
          "ID-JW" => "Jawa",
          "ID-KA" => "Kalimantan",
          "ID-ML" => "Maluku",
          "ID-NU" => "Nusa Tenggara",
          "ID-PP" => "Papua",
          "ID-SL" => "Sulawesi",
          "ID-SM" => "Sumatera"
        },
        "IE" => {
          "IE-C" => "Connaught",
          "IE-L" => "Leinster",
          "IE-M" => "Munster",
          "IE-U" => "Ulster"
        },
        "IL" => {
          "IL-D" => "HaDarom",
          "IL-M" => "HaMerkaz",
          "IL-Z" => "HaTsafon",
          "IL-HA" => "H̱efa",
          "IL-TA" => "Tel Aviv",
          "IL-JM" => "Yerushalayim"
        },
        "IN" => {
          "IN-AP" => "Andhra Pradesh",
          "IN-AR" => "Arunāchal Pradesh",
          "IN-AS" => "Assam",
          "IN-BR" => "Bihār",
          "IN-CT" => "Chhattīsgarh[note 1]",
          "IN-GA" => "Goa",
          "IN-GJ" => "Gujarāt",
          "IN-HR" => "Haryāna",
          "IN-HP" => "Himāchal Pradesh",
          "IN-JH" => "Jhārkhand",
          "IN-KA" => "Karnātaka",
          "IN-KL" => "Kerala",
          "IN-MP" => "Madhya Pradesh",
          "IN-MH" => "Mahārāshtra",
          "IN-MN" => "Manipur",
          "IN-ML" => "Meghālaya",
          "IN-MZ" => "Mizoram",
          "IN-NL" => "Nāgāland",
          "IN-OR" => "Odisha[note 2]",
          "IN-PB" => "Punjab",
          "IN-RJ" => "Rājasthān",
          "IN-SK" => "Sikkim",
          "IN-TN" => "Tamil Nādu",
          "IN-TG" => "Telangāna[note 3]",
          "IN-TR" => "Tripura",
          "IN-UT" => "Uttarākhand[note 4]",
          "IN-UP" => "Uttar Pradesh",
          "IN-WB" => "West Bengal",
          "IN-AN" => "Andaman and Nicobar Islands",
          "IN-CH" => "Chandigarh",
          "IN-DH" => "Dādra and Nagar Haveli and Damān and Diu[note 5]",
          "IN-DL" => "Delhi",
          "IN-JK" => "Jammu and Kashmīr",
          "IN-LA" => "Ladākh",
          "IN-LD" => "Lakshadweep",
          "IN-PY" => "Puducherry"
        },
        "IQ" => {
          "IQ-AN" => "Al Anbār",
          "IQ-BA" => "Al Başrah",
          "IQ-MU" => "Al Muthanná",
          "IQ-QA" => "Al Qādisīyah",
          "IQ-NA" => "An Najaf",
          "IQ-AR" => "Arbīl",
          "IQ-SU" => "As Sulaymānīyah",
          "IQ-BB" => "Bābil",
          "IQ-BG" => "Baghdād",
          "IQ-DA" => "Dahūk",
          "IQ-DQ" => "Dhī Qār",
          "IQ-DI" => "Diyālá",
          "IQ-KA" => "Karbalā’",
          "IQ-KI" => "Kirkūk",
          "IQ-MA" => "Maysān",
          "IQ-NI" => "Nīnawá",
          "IQ-SD" => "Şalāḩ ad Dīn",
          "IQ-WA" => "Wāsiţ",
          "IQ-HA" => "Halabja"
        },
        "IR" => {
          "IR-30" => "Alborz",
          "IR-24" => "Ardabīl",
          "IR-04" => "Āz̄ārbāyjān-e Ghārbī",
          "IR-03" => "Āz̄ārbāyjān-e Shārqī",
          "IR-18" => "Būshehr",
          "IR-14" => "Chahār Maḩāl va Bakhtīārī",
          "IR-10" => "Eşfahān",
          "IR-07" => "Fārs",
          "IR-01" => "Gīlān",
          "IR-27" => "Golestān",
          "IR-13" => "Hamadān",
          "IR-22" => "Hormozgān",
          "IR-16" => "Īlām",
          "IR-08" => "Kermān",
          "IR-05" => "Kermānshāh",
          "IR-29" => "Khorāsān-e Jonūbī",
          "IR-09" => "Khorāsān-e Raẕavī",
          "IR-28" => "Khorāsān-e Shomālī",
          "IR-06" => "Khūzestān",
          "IR-17" => "Kohgīlūyeh va Bowyer Aḩmad",
          "IR-12" => "Kordestān",
          "IR-15" => "Lorestān",
          "IR-00" => "Markazī",
          "IR-02" => "Māzandarān",
          "IR-26" => "Qazvīn",
          "IR-25" => "Qom",
          "IR-20" => "Semnān",
          "IR-11" => "Sīstān va Balūchestān",
          "IR-23" => "Tehrān",
          "IR-21" => "Yazd",
          "IR-19" => "Zanjān"
        },
        "IS" => {
          "IS-1" => "Höfuðborgarsvæði",
          "IS-2" => "Suðurnes",
          "IS-3" => "Vesturland",
          "IS-4" => "Vestfirðir",
          "IS-5" => "Norðurland vestra",
          "IS-6" => "Norðurland eystra",
          "IS-7" => "Austurland",
          "IS-8" => "Suðurland"
        },
        "IT" => {
          "IT-65" => "Abruzzo",
          "IT-77" => "Basilicata",
          "IT-78" => "Calabria",
          "IT-72" => "Campania",
          "IT-45" => "Emilia-Romagna",
          "IT-62" => "Lazio",
          "IT-42" => "Liguria",
          "IT-25" => "Lombardia",
          "IT-57" => "Marche",
          "IT-67" => "Molise",
          "IT-21" => "Piemonte",
          "IT-75" => "Puglia",
          "IT-52" => "Toscana",
          "IT-55" => "Umbria",
          "IT-34" => "Veneto"
        },
        "JM" => {
          "JM-13" => "Clarendon",
          "JM-09" => "Hanover",
          "JM-01" => "Kingston",
          "JM-12" => "Manchester",
          "JM-04" => "Portland",
          "JM-02" => "Saint Andrew",
          "JM-06" => "Saint Ann",
          "JM-14" => "Saint Catherine",
          "JM-11" => "Saint Elizabeth",
          "JM-08" => "Saint James",
          "JM-05" => "Saint Mary",
          "JM-03" => "Saint Thomas",
          "JM-07" => "Trelawny",
          "JM-10" => "Westmoreland"
        },
        "JO" => {
          "JO-AJ" => "‘Ajlūn",
          "JO-AQ" => "Al ‘Aqabah",
          "JO-AM" => "Al ‘A̅şimah",
          "JO-BA" => "Al Balqā’",
          "JO-KA" => "Al Karak",
          "JO-MA" => "Al Mafraq",
          "JO-AT" => "Aţ Ţafīlah",
          "JO-AZ" => "Az Zarqā’",
          "JO-IR" => "Irbid",
          "JO-JA" => "Jarash",
          "JO-MN" => "Ma‘ān",
          "JO-MD" => "Mādabā"
        },
        "JP" => {
          "JP-23" => "Aiti",
          "JP-05" => "Akita",
          "JP-02" => "Aomori",
          "JP-38" => "Ehime",
          "JP-21" => "Gihu",
          "JP-10" => "Gunma",
          "JP-34" => "Hirosima",
          "JP-01" => "Hokkaidô",
          "JP-18" => "Hukui",
          "JP-40" => "Hukuoka",
          "JP-07" => "Hukusima",
          "JP-28" => "Hyôgo",
          "JP-08" => "Ibaraki",
          "JP-17" => "Isikawa",
          "JP-03" => "Iwate",
          "JP-37" => "Kagawa",
          "JP-46" => "Kagosima",
          "JP-14" => "Kanagawa",
          "JP-39" => "Kôti",
          "JP-43" => "Kumamoto",
          "JP-26" => "Kyôto",
          "JP-24" => "Mie",
          "JP-04" => "Miyagi",
          "JP-45" => "Miyazaki",
          "JP-20" => "Nagano",
          "JP-42" => "Nagasaki",
          "JP-29" => "Nara",
          "JP-15" => "Niigata",
          "JP-44" => "Ôita",
          "JP-33" => "Okayama",
          "JP-47" => "Okinawa",
          "JP-27" => "Ôsaka",
          "JP-41" => "Saga",
          "JP-11" => "Saitama",
          "JP-25" => "Siga",
          "JP-32" => "Simane",
          "JP-22" => "Sizuoka",
          "JP-12" => "Tiba",
          "JP-09" => "Totigi",
          "JP-36" => "Tokusima",
          "JP-13" => "Tôkyô",
          "JP-31" => "Tottori",
          "JP-16" => "Toyama",
          "JP-30" => "Wakayama",
          "JP-06" => "Yamagata",
          "JP-35" => "Yamaguti",
          "JP-19" => "Yamanasi"
        },
        "KE" => {
          "KE-01" => "Baringo",
          "KE-02" => "Bomet",
          "KE-03" => "Bungoma",
          "KE-04" => "Busia",
          "KE-05" => "Elgeyo/Marakwet",
          "KE-06" => "Embu",
          "KE-07" => "Garissa",
          "KE-08" => "Homa Bay",
          "KE-09" => "Isiolo",
          "KE-10" => "Kajiado",
          "KE-11" => "Kakamega",
          "KE-12" => "Kericho",
          "KE-13" => "Kiambu",
          "KE-14" => "Kilifi",
          "KE-15" => "Kirinyaga",
          "KE-16" => "Kisii",
          "KE-17" => "Kisumu",
          "KE-18" => "Kitui",
          "KE-19" => "Kwale",
          "KE-20" => "Laikipia",
          "KE-21" => "Lamu",
          "KE-22" => "Machakos",
          "KE-23" => "Makueni",
          "KE-24" => "Mandera",
          "KE-25" => "Marsabit",
          "KE-26" => "Meru",
          "KE-27" => "Migori",
          "KE-28" => "Mombasa",
          "KE-29" => "Murang'a",
          "KE-30" => "Nairobi City",
          "KE-31" => "Nakuru",
          "KE-32" => "Nandi",
          "KE-33" => "Narok",
          "KE-34" => "Nyamira",
          "KE-35" => "Nyandarua",
          "KE-36" => "Nyeri",
          "KE-37" => "Samburu",
          "KE-38" => "Siaya",
          "KE-39" => "Taita/Taveta",
          "KE-40" => "Tana River",
          "KE-41" => "Tharaka-Nithi",
          "KE-42" => "Trans Nzoia",
          "KE-43" => "Turkana",
          "KE-44" => "Uasin Gishu",
          "KE-45" => "Vihiga",
          "KE-46" => "Wajir",
          "KE-47" => "West Pokot"
        },
        "KG" => {
          "KG-GB" => "Bishkek Shaary",
          "KG-GO" => "Osh Shaary",
          "KG-B" => "Batken",
          "KG-C" => "Chüy",
          "KG-J" => "Jalal-Abad",
          "KG-N" => "Naryn",
          "KG-O" => "Osh",
          "KG-T" => "Talas",
          "KG-Y" => "Ysyk-Köl"
        },
        "KH" => {
          "KH-12" => "Phnom Penh",
          "KH-2" => "Baat Dambang",
          "KH-1" => "Banteay Mean Choăy",
          "KH-3" => "Kampong Chaam",
          "KH-4" => "Kampong Chhnang",
          "KH-5" => "Kampong Spueu",
          "KH-6" => "Kampong Thum",
          "KH-7" => "Kampot",
          "KH-8" => "Kandaal",
          "KH-9" => "Kaoh Kong",
          "KH-10" => "Kracheh",
          "KH-23" => "Kaeb",
          "KH-24" => "Pailin",
          "KH-18" => "Preah Sihanouk",
          "KH-11" => "Mondol Kiri",
          "KH-22" => "Otdar Mean Chey",
          "KH-15" => "Pousaat",
          "KH-13" => "Preah Vihear",
          "KH-14" => "Prey Veaeng",
          "KH-16" => "Rotanak Kiri",
          "KH-17" => "Siem Reab",
          "KH-19" => "Stueng Traeng",
          "KH-20" => "Svaay Rieng",
          "KH-21" => "Taakaev",
          "KH-25" => "Tbong Khmum"
        },
        "KI" => {
          "KI-G" => "Gilbert Islands",
          "KI-L" => "Line Islands",
          "KI-P" => "Phoenix Islands"
        },
        "KM" => {
          "KM-G" => "Grande Comore",
          "KM-A" => "Anjouan",
          "KM-M" => "Mohéli"
        },
        "KN" => {
          "KN-K" => "Saint Kitts",
          "KN-N" => "Nevis"
        },
        "KP" => {
          "KP-01" => "Pyongyang",
          "KP-13" => "Rason",
          "KP-14" => "Nampo",
          "KP-02" => "South Pyongan",
          "KP-03" => "North Pyongan",
          "KP-04" => "Chagang",
          "KP-05" => "South Hwanghae",
          "KP-06" => "North Hwanghae",
          "KP-07" => "Kangwon",
          "KP-08" => "South Hamgyong",
          "KP-09" => "North Hamgyong",
          "KP-10" => "Ryanggang"
        },
        "KR" => {
          "KR-11" => "Seoul",
          "KR-26" => "Busan",
          "KR-27" => "Daegu",
          "KR-30" => "Daejeon",
          "KR-29" => "Gwangju",
          "KR-28" => "Incheon",
          "KR-31" => "Ulsan",
          "KR-43" => "North Chungcheong",
          "KR-44" => "South Chungcheong",
          "KR-42" => "Gangwon",
          "KR-41" => "Gyeonggi",
          "KR-47" => "North Gyeongsang",
          "KR-48" => "South Gyeongsang",
          "KR-45" => "North Jeolla",
          "KR-46" => "South Jeolla",
          "KR-49" => "Jeju",
          "KR-50" => "Sejong"
        },
        "KW" => {
          "KW-AH" => "Al Aḩmadī",
          "KW-FA" => "Al Farwānīyah",
          "KW-JA" => "Al Jahrā’",
          "KW-KU" => "Al ‘Āşimah",
          "KW-HA" => "Ḩawallī",
          "KW-MU" => "Mubārak al Kabīr"
        },
        "KZ" => {
          "KZ-ALA" => "Almaty",
          "KZ-ALM" => "Almaty oblysy",
          "KZ-AKM" => "Aqmola oblysy",
          "KZ-AKT" => "Aqtöbe oblysy",
          "KZ-ATY" => "Atyraū oblysy",
          "KZ-ZAP" => "Batys Qazaqstan oblysy",
          "KZ-MAN" => "Mangghystaū oblysy",
          "KZ-AST" => "Nur-Sultan",
          "KZ-PAV" => "Pavlodar oblysy",
          "KZ-KAR" => "Qaraghandy oblysy",
          "KZ-KUS" => "Qostanay oblysy",
          "KZ-KZY" => "Qyzylorda oblysy",
          "KZ-VOS" => "Shyghys Qazaqstan oblysy",
          "KZ-SHY" => "Shymkent",
          "KZ-SEV" => "Soltüstik Qazaqstan oblysy",
          "KZ-YUZ" => "Türkistan oblysy",
          "KZ-ZHA" => "Zhambyl oblysy"
        },
        "LA" => {
          "LA-VT" => "Viangchan",
          "LA-AT" => "Attapu",
          "LA-BK" => "Bokèo",
          "LA-BL" => "Bolikhamxai",
          "LA-CH" => "Champasak",
          "LA-HO" => "Houaphan",
          "LA-KH" => "Khammouan",
          "LA-LM" => "Louang Namtha",
          "LA-LP" => "Louangphabang",
          "LA-OU" => "Oudômxai",
          "LA-PH" => "Phôngsali",
          "LA-SL" => "Salavan",
          "LA-SV" => "Savannakhét",
          "LA-VI" => "Viangchan",
          "LA-XA" => "Xaignabouli",
          "LA-XE" => "Xékong",
          "LA-XI" => "Xiangkhouang",
          "LA-XS" => "Xaisômboun"
        },
        "LB" => {
          "LB-AK" => "Aakkâr",
          "LB-BH" => "Baalbek-Hermel",
          "LB-BI" => "Béqaa",
          "LB-BA" => "Beyrouth",
          "LB-AS" => "Liban-Nord",
          "LB-JA" => "Liban-Sud",
          "LB-JL" => "Mont-Liban",
          "LB-NA" => "Nabatîyé"
        },
        "LC" => {
          "LC-01" => "Anse la Raye",
          "LC-12" => "Canaries",
          "LC-02" => "Castries",
          "LC-03" => "Choiseul",
          "LC-05" => "Dennery",
          "LC-06" => "Gros Islet",
          "LC-07" => "Laborie",
          "LC-08" => "Micoud",
          "LC-10" => "Soufrière",
          "LC-11" => "Vieux Fort"
        },
        "LI" => {
          "LI-01" => "Balzers",
          "LI-02" => "Eschen",
          "LI-03" => "Gamprin",
          "LI-04" => "Mauren",
          "LI-05" => "Planken",
          "LI-06" => "Ruggell",
          "LI-07" => "Schaan",
          "LI-08" => "Schellenberg",
          "LI-09" => "Triesen",
          "LI-10" => "Triesenberg",
          "LI-11" => "Vaduz"
        },
        "LK" => {
          "LK-1" => "Western Province",
          "LK-2" => "Central Province",
          "LK-3" => "Southern Province",
          "LK-4" => "Northern Province",
          "LK-5" => "Eastern Province",
          "LK-6" => "North Western Province",
          "LK-7" => "North Central Province",
          "LK-8" => "Uva Province",
          "LK-9" => "Sabaragamuwa Province"
        },
        "LR" => {
          "LR-BM" => "Bomi",
          "LR-BG" => "Bong",
          "LR-GP" => "Gbarpolu",
          "LR-GB" => "Grand Bassa",
          "LR-CM" => "Grand Cape Mount",
          "LR-GG" => "Grand Gedeh",
          "LR-GK" => "Grand Kru",
          "LR-LO" => "Lofa",
          "LR-MG" => "Margibi",
          "LR-MY" => "Maryland",
          "LR-MO" => "Montserrado",
          "LR-NI" => "Nimba",
          "LR-RI" => "River Cess(local variant is Rivercess)",
          "LR-RG" => "River Gee",
          "LR-SI" => "Sinoe"
        },
        "LS" => {
          "LS-D" => "Berea",
          "LS-B" => "Botha-Bothe",
          "LS-C" => "Leribe",
          "LS-E" => "Mafeteng",
          "LS-A" => "Maseru",
          "LS-F" => "Mohale's Hoek",
          "LS-J" => "Mokhotlong",
          "LS-H" => "Qacha's Nek",
          "LS-G" => "Quthing",
          "LS-K" => "Thaba-Tseka"
        },
        "LT" => {
          "LT-AL" => "Alytaus apskritis",
          "LT-KU" => "Kauno apskritis",
          "LT-KL" => "Klaipėdos apskritis",
          "LT-MR" => "Marijampolės apskritis",
          "LT-PN" => "Panevėžio apskritis",
          "LT-SA" => "Šiaulių apskritis",
          "LT-TA" => "Tauragės apskritis",
          "LT-TE" => "Telšių apskritis",
          "LT-UT" => "Utenos apskritis",
          "LT-VL" => "Vilniaus apskritis"
        },
        "LU" => {
          "LU-CA" => "Capellen",
          "LU-CL" => "Clervaux",
          "LU-DI" => "Diekirch",
          "LU-EC" => "Echternach",
          "LU-ES" => "Esch-sur-Alzette",
          "LU-GR" => "Grevenmacher",
          "LU-LU" => "Luxembourg",
          "LU-ME" => "Mersch",
          "LU-RD" => "Redange",
          "LU-RM" => "Remich",
          "LU-VD" => "Vianden",
          "LU-WI" => "Wiltz"
        },
        "LV" => {
          "LV-001" => "Aglonas novads",
          "LV-002" => "Aizkraukles novads",
          "LV-003" => "Aizputes novads",
          "LV-004" => "Aknīstes novads",
          "LV-005" => "Alojas novads",
          "LV-006" => "Alsungas novads",
          "LV-007" => "Alūksnes novads",
          "LV-008" => "Amatas novads",
          "LV-009" => "Apes novads",
          "LV-010" => "Auces novads",
          "LV-011" => "Ādažu novads",
          "LV-012" => "Babītes novads",
          "LV-013" => "Baldones novads",
          "LV-014" => "Baltinavas novads",
          "LV-015" => "Balvu novads",
          "LV-016" => "Bauskas novads",
          "LV-017" => "Beverīnas novads",
          "LV-018" => "Brocēnu novads",
          "LV-019" => "Burtnieku novads",
          "LV-020" => "Carnikavas novads",
          "LV-021" => "Cesvaines novads",
          "LV-022" => "Cēsu novads",
          "LV-023" => "Ciblas novads",
          "LV-024" => "Dagdas novads",
          "LV-025" => "Daugavpils novads",
          "LV-026" => "Dobeles novads",
          "LV-027" => "Dundagas novads",
          "LV-028" => "Durbes novads",
          "LV-029" => "Engures novads",
          "LV-030" => "Ērgļu novads",
          "LV-031" => "Garkalnes novads",
          "LV-032" => "Grobiņas novads",
          "LV-033" => "Gulbenes novads",
          "LV-034" => "Iecavas novads",
          "LV-035" => "Ikšķiles novads",
          "LV-036" => "Ilūkstes novads",
          "LV-037" => "Inčukalna novads",
          "LV-038" => "Jaunjelgavas novads",
          "LV-039" => "Jaunpiebalgas novads",
          "LV-040" => "Jaunpils novads",
          "LV-041" => "Jelgavas novads",
          "LV-042" => "Jēkabpils novads",
          "LV-043" => "Kandavas novads",
          "LV-044" => "Kārsavas novads",
          "LV-045" => "Kocēnu novads",
          "LV-046" => "Kokneses novads",
          "LV-047" => "Krāslavas novads",
          "LV-048" => "Krimuldas novads",
          "LV-049" => "Krustpils novads",
          "LV-050" => "Kuldīgas novads",
          "LV-051" => "Ķeguma novads",
          "LV-052" => "Ķekavas novads",
          "LV-053" => "Lielvārdes novads",
          "LV-054" => "Limbažu novads",
          "LV-055" => "Līgatnes novads",
          "LV-056" => "Līvānu novads",
          "LV-057" => "Lubānas novads",
          "LV-058" => "Ludzas novads",
          "LV-059" => "Madonas novads",
          "LV-060" => "Mazsalacas novads",
          "LV-061" => "Mālpils novads",
          "LV-062" => "Mārupes novads",
          "LV-063" => "Mērsraga novads",
          "LV-064" => "Naukšēnu novads",
          "LV-065" => "Neretas novads",
          "LV-066" => "Nīcas novads",
          "LV-067" => "Ogres novads",
          "LV-068" => "Olaines novads",
          "LV-069" => "Ozolnieku novads",
          "LV-070" => "Pārgaujas novads",
          "LV-071" => "Pāvilostas novads",
          "LV-072" => "Pļaviņu novads",
          "LV-073" => "Preiļu novads",
          "LV-074" => "Priekules novads",
          "LV-075" => "Priekuļu novads",
          "LV-076" => "Raunas novads",
          "LV-077" => "Rēzeknes novads",
          "LV-078" => "Riebiņu novads",
          "LV-079" => "Rojas novads",
          "LV-080" => "Ropažu novads",
          "LV-081" => "Rucavas novads",
          "LV-082" => "Rugāju novads",
          "LV-083" => "Rundāles novads",
          "LV-084" => "Rūjienas novads",
          "LV-085" => "Salas novads",
          "LV-086" => "Salacgrīvas novads",
          "LV-087" => "Salaspils novads",
          "LV-088" => "Saldus novads",
          "LV-089" => "Saulkrastu novads",
          "LV-090" => "Sējas novads",
          "LV-091" => "Siguldas novads",
          "LV-092" => "Skrīveru novads",
          "LV-093" => "Skrundas novads",
          "LV-094" => "Smiltenes novads",
          "LV-095" => "Stopiņu novads",
          "LV-096" => "Strenču novads",
          "LV-097" => "Talsu novads",
          "LV-098" => "Tērvetes novads",
          "LV-099" => "Tukuma novads",
          "LV-100" => "Vaiņodes novads",
          "LV-101" => "Valkas novads",
          "LV-102" => "Varakļānu novads",
          "LV-103" => "Vārkavas novads",
          "LV-104" => "Vecpiebalgas novads",
          "LV-105" => "Vecumnieku novads",
          "LV-106" => "Ventspils novads",
          "LV-107" => "Viesītes novads",
          "LV-108" => "Viļakas novads",
          "LV-109" => "Viļānu novads",
          "LV-110" => "Zilupes novads",
          "LV-DGV" => "Daugavpils",
          "LV-JEL" => "Jelgava",
          "LV-JKB" => "Jēkabpils",
          "LV-JUR" => "Jūrmala",
          "LV-LPX" => "Liepāja",
          "LV-REZ" => "Rēzekne",
          "LV-RIX" => "Rīga",
          "LV-VMR" => "Valmiera",
          "LV-VEN" => "Ventspils"
        },
        "LY" => {
          "LY-BU" => "Al Buţnān",
          "LY-JA" => "Al Jabal al Akhḑar",
          "LY-JG" => "Al Jabal al Gharbī",
          "LY-JI" => "Al Jafārah",
          "LY-JU" => "Al Jufrah",
          "LY-KF" => "Al Kufrah",
          "LY-MJ" => "Al Marj",
          "LY-MB" => "Al Marqab",
          "LY-WA" => "Al Wāḩāt",
          "LY-NQ" => "An Nuqāţ al Khams",
          "LY-ZA" => "Az Zāwiyah",
          "LY-BA" => "Banghāzī",
          "LY-DR" => "Darnah",
          "LY-GT" => "Ghāt",
          "LY-MI" => "Mişrātah",
          "LY-MQ" => "Murzuq",
          "LY-NL" => "Nālūt",
          "LY-SB" => "Sabhā",
          "LY-SR" => "Surt",
          "LY-TB" => "Ţarābulus",
          "LY-WD" => "Wādī al Ḩayāt",
          "LY-WS" => "Wādī ash Shāţi’"
        },
        "MA" => {
          "MA-05" => "Béni Mellal-Khénifra",
          "MA-06" => "Casablanca-Settat",
          "MA-12" => "Dakhla-Oued Ed-Dahab (EH)",
          "MA-08" => "Drâa-Tafilalet",
          "MA-03" => "Fès-Meknès",
          "MA-10" => "Guelmim-Oued Noun (EH-partial)",
          "MA-11" => "Laâyoune-Sakia El Hamra (EH-partial)",
          "MA-07" => "Marrakech-Safi",
          "MA-02" => "L'Oriental",
          "MA-04" => "Rabat-Salé-Kénitra",
          "MA-09" => "Souss-Massa",
          "MA-01" => "Tanger-Tétouan-Al Hoceïma"
        },
        "MC" => {
          "MC-FO" => "Fontvieille",
          "MC-JE" => "Jardin Exotique",
          "MC-CL" => "La Colle",
          "MC-CO" => "La Condamine",
          "MC-GA" => "La Gare",
          "MC-SO" => "La Source",
          "MC-LA" => "Larvotto",
          "MC-MA" => "Malbousquet",
          "MC-MO" => "Monaco-Ville",
          "MC-MG" => "Moneghetti",
          "MC-MC" => "Monte-Carlo",
          "MC-MU" => "Moulins",
          "MC-PH" => "Port-Hercule",
          "MC-SR" => "Saint-Roman",
          "MC-SD" => "Sainte-Dévote",
          "MC-SP" => "Spélugues",
          "MC-VR" => "Vallon de la Rousse"
        },
        "MD" => {
          "MD-GA" => "Găgăuzia, Unitatea teritorială autonomă (UTAG)",
          "MD-BA" => "Bălți",
          "MD-BD" => "Bender [Tighina]",
          "MD-CU" => "Chișinău",
          "MD-AN" => "Anenii Noi",
          "MD-BS" => "Basarabeasca",
          "MD-BR" => "Briceni",
          "MD-CA" => "Cahul",
          "MD-CT" => "Cantemir",
          "MD-CL" => "Călărași",
          "MD-CS" => "Căușeni",
          "MD-CM" => "Cimișlia",
          "MD-CR" => "Criuleni",
          "MD-DO" => "Dondușeni",
          "MD-DR" => "Drochia",
          "MD-DU" => "Dubăsari",
          "MD-ED" => "Edineț",
          "MD-FA" => "Fălești",
          "MD-FL" => "Florești",
          "MD-GL" => "Glodeni",
          "MD-HI" => "Hîncești",
          "MD-IA" => "Ialoveni",
          "MD-LE" => "Leova",
          "MD-NI" => "Nisporeni",
          "MD-OC" => "Ocnița",
          "MD-OR" => "Orhei",
          "MD-RE" => "Rezina",
          "MD-RI" => "Rîșcani",
          "MD-SI" => "Sîngerei",
          "MD-SO" => "Soroca",
          "MD-ST" => "Strășeni",
          "MD-SD" => "Șoldănești",
          "MD-SV" => "Ștefan Vodă",
          "MD-TA" => "Taraclia",
          "MD-TE" => "Telenești",
          "MD-UN" => "Ungheni",
          "MD-SN" => "Stînga Nistrului, unitatea teritorială din"
        },
        "ME" => {
          "ME-01" => "Andrijevica",
          "ME-02" => "Bar",
          "ME-03" => "Berane",
          "ME-04" => "Bijelo Polje",
          "ME-05" => "Budva",
          "ME-06" => "Cetinje",
          "ME-07" => "Danilovgrad",
          "ME-08" => "Herceg-Novi",
          "ME-09" => "Kolašin",
          "ME-10" => "Kotor",
          "ME-11" => "Mojkovac",
          "ME-12" => "Nikšić",
          "ME-13" => "Plav",
          "ME-14" => "Pljevlja",
          "ME-15" => "Plužine",
          "ME-16" => "Podgorica",
          "ME-17" => "Rožaje",
          "ME-18" => "Šavnik",
          "ME-19" => "Tivat",
          "ME-20" => "Ulcinj",
          "ME-21" => "Žabljak",
          "ME-22" => "Gusinje",
          "ME-23" => "Petnjica",
          "ME-24" => "Tuzi"
        },
        "MG" => {
          "MG-T" => "Antananarivo",
          "MG-D" => "Antsiranana",
          "MG-F" => "Fianarantsoa",
          "MG-M" => "Mahajanga",
          "MG-A" => "Toamasina",
          "MG-U" => "Toliara"
        },
        "MH" => {
          "MH-L" => "Ralik chain",
          "MH-T" => "Ratak chain"
        },
        "MK" => {
          "MK-801" => "Aerodrom",
          "MK-802" => "Aračinovo",
          "MK-201" => "Berovo",
          "MK-501" => "Bitola",
          "MK-401" => "Bogdanci",
          "MK-601" => "Bogovinje",
          "MK-402" => "Bosilovo",
          "MK-602" => "Brvenica",
          "MK-803" => "Butel",
          "MK-814" => "Centar",
          "MK-313" => "Centar Župa",
          "MK-815" => "Čair",
          "MK-109" => "Čaška",
          "MK-210" => "Češinovo-Obleševo",
          "MK-816" => "Čučer-Sandevo",
          "MK-303" => "Debar",
          "MK-304" => "Debarca",
          "MK-203" => "Delčevo",
          "MK-502" => "Demir Hisar",
          "MK-103" => "Demir Kapija",
          "MK-406" => "Dojran",
          "MK-503" => "Dolneni",
          "MK-804" => "Gazi Baba",
          "MK-405" => "Gevgelija",
          "MK-805" => "Gjorče Petrov",
          "MK-604" => "Gostivar",
          "MK-102" => "Gradsko",
          "MK-807" => "Ilinden",
          "MK-606" => "Jegunovce",
          "MK-205" => "Karbinci",
          "MK-808" => "Karpoš",
          "MK-104" => "Kavadarci",
          "MK-307" => "Kičevo",
          "MK-809" => "Kisela Voda",
          "MK-206" => "Kočani",
          "MK-407" => "Konče",
          "MK-701" => "Kratovo",
          "MK-702" => "Kriva Palanka",
          "MK-504" => "Krivogaštani",
          "MK-505" => "Kruševo",
          "MK-703" => "Kumanovo",
          "MK-704" => "Lipkovo",
          "MK-105" => "Lozovo",
          "MK-207" => "Makedonska Kamenica",
          "MK-308" => "Makedonski Brod",
          "MK-607" => "Mavrovo i Rostuše",
          "MK-506" => "Mogila",
          "MK-106" => "Negotino",
          "MK-507" => "Novaci",
          "MK-408" => "Novo Selo",
          "MK-310" => "Ohrid",
          "MK-208" => "Pehčevo",
          "MK-810" => "Petrovec",
          "MK-311" => "Plasnica",
          "MK-508" => "Prilep",
          "MK-209" => "Probištip",
          "MK-409" => "Radoviš",
          "MK-705" => "Rankovce",
          "MK-509" => "Resen",
          "MK-107" => "Rosoman",
          "MK-811" => "Saraj",
          "MK-812" => "Sopište",
          "MK-706" => "Staro Nagoričane",
          "MK-312" => "Struga",
          "MK-410" => "Strumica",
          "MK-813" => "Studeničani",
          "MK-108" => "Sveti Nikole",
          "MK-211" => "Štip",
          "MK-817" => "Šuto Orizari",
          "MK-608" => "Tearce",
          "MK-609" => "Tetovo",
          "MK-403" => "Valandovo",
          "MK-404" => "Vasilevo",
          "MK-101" => "Veles",
          "MK-301" => "Vevčani",
          "MK-202" => "Vinica",
          "MK-603" => "Vrapčište",
          "MK-806" => "Zelenikovo",
          "MK-204" => "Zrnovci",
          "MK-605" => "Želino"
        },
        "ML" => {
          "ML-BKO" => "Bamako",
          "ML-7" => "Gao",
          "ML-1" => "Kayes",
          "ML-8" => "Kidal",
          "ML-2" => "Koulikoro",
          "ML-9" => "Ménaka",
          "ML-5" => "Mopti",
          "ML-4" => "Ségou",
          "ML-3" => "Sikasso",
          "ML-10" => "Taoudénit(local variants are Taoudenni, Taoudéni)",
          "ML-6" => "Tombouctou"
        },
        "MM" => {
          "MM-07" => "Ayeyarwady",
          "MM-02" => "Bago",
          "MM-03" => "Magway",
          "MM-04" => "Mandalay",
          "MM-01" => "Sagaing",
          "MM-05" => "Tanintharyi",
          "MM-06" => "Yangon",
          "MM-14" => "Chin",
          "MM-11" => "Kachin",
          "MM-12" => "Kayah",
          "MM-13" => "Kayin",
          "MM-15" => "Mon",
          "MM-16" => "Rakhine",
          "MM-17" => "Shan",
          "MM-18" => "Nay Pyi Taw"
        },
        "MN" => {
          "MN-1" => "Ulaanbaatar",
          "MN-073" => "Arhangay",
          "MN-069" => "Bayanhongor",
          "MN-071" => "Bayan-Ölgiy",
          "MN-067" => "Bulgan",
          "MN-037" => "Darhan uul",
          "MN-061" => "Dornod",
          "MN-063" => "Dornogovĭ",
          "MN-059" => "Dundgovĭ",
          "MN-057" => "Dzavhan",
          "MN-065" => "Govĭ-Altay",
          "MN-064" => "Govĭ-Sümber",
          "MN-039" => "Hentiy",
          "MN-043" => "Hovd",
          "MN-041" => "Hövsgöl",
          "MN-053" => "Ömnögovĭ",
          "MN-035" => "Orhon",
          "MN-055" => "Övörhangay",
          "MN-049" => "Selenge",
          "MN-051" => "Sühbaatar",
          "MN-047" => "Töv",
          "MN-046" => "Uvs"
        },
        "MR" => {
          "MR-07" => "Adrar",
          "MR-03" => "Assaba",
          "MR-05" => "Brakna",
          "MR-08" => "Dakhlet Nouâdhibou",
          "MR-04" => "Gorgol",
          "MR-10" => "Guidimaka",
          "MR-01" => "Hodh ech Chargui",
          "MR-02" => "Hodh el Gharbi",
          "MR-12" => "Inchiri",
          "MR-09" => "Tagant",
          "MR-11" => "Tiris Zemmour",
          "MR-06" => "Trarza"
        },
        "MT" => {
          "MT-01" => "Attard",
          "MT-02" => "Balzan",
          "MT-03" => "Birgu",
          "MT-04" => "Birkirkara",
          "MT-05" => "Birżebbuġa",
          "MT-06" => "Bormla",
          "MT-07" => "Dingli",
          "MT-08" => "Fgura",
          "MT-09" => "Floriana",
          "MT-10" => "Fontana",
          "MT-11" => "Gudja",
          "MT-12" => "Gżira",
          "MT-13" => "Għajnsielem",
          "MT-14" => "Għarb",
          "MT-15" => "Għargħur",
          "MT-16" => "Għasri",
          "MT-17" => "Għaxaq",
          "MT-18" => "Ħamrun",
          "MT-19" => "Iklin",
          "MT-20" => "Isla",
          "MT-21" => "Kalkara",
          "MT-22" => "Kerċem",
          "MT-23" => "Kirkop",
          "MT-24" => "Lija",
          "MT-25" => "Luqa",
          "MT-26" => "Marsa",
          "MT-27" => "Marsaskala",
          "MT-28" => "Marsaxlokk",
          "MT-29" => "Mdina",
          "MT-30" => "Mellieħa",
          "MT-31" => "Mġarr",
          "MT-32" => "Mosta",
          "MT-33" => "Mqabba",
          "MT-34" => "Msida",
          "MT-35" => "Mtarfa",
          "MT-36" => "Munxar",
          "MT-37" => "Nadur",
          "MT-38" => "Naxxar",
          "MT-39" => "Paola",
          "MT-40" => "Pembroke",
          "MT-41" => "Pietà",
          "MT-42" => "Qala",
          "MT-43" => "Qormi",
          "MT-44" => "Qrendi",
          "MT-45" => "Rabat Gozo",
          "MT-46" => "Rabat Malta",
          "MT-47" => "Safi",
          "MT-48" => "Saint Julian's",
          "MT-49" => "Saint John",
          "MT-50" => "Saint Lawrence",
          "MT-51" => "Saint Paul's Bay",
          "MT-52" => "Sannat",
          "MT-53" => "Saint Lucia's",
          "MT-54" => "Santa Venera",
          "MT-55" => "Siġġiewi",
          "MT-56" => "Sliema",
          "MT-57" => "Swieqi",
          "MT-58" => "Ta' Xbiex",
          "MT-59" => "Tarxien",
          "MT-60" => "Valletta",
          "MT-61" => "Xagħra",
          "MT-62" => "Xewkija",
          "MT-63" => "Xgħajra",
          "MT-64" => "Żabbar",
          "MT-65" => "Żebbuġ Gozo",
          "MT-66" => "Żebbuġ Malta",
          "MT-67" => "Żejtun",
          "MT-68" => "Żurrieq"
        },
        "MU" => {
          "MU-AG" => "Agalega Islands",
          "MU-CC" => "Cargados Carajos Shoals(local variant is Saint Brandon Islands)",
          "MU-RO" => "Rodrigues Island",
          "MU-BL" => "Black River",
          "MU-FL" => "Flacq",
          "MU-GP" => "Grand Port",
          "MU-MO" => "Moka",
          "MU-PA" => "Pamplemousses",
          "MU-PW" => "Plaines Wilhems",
          "MU-PL" => "Port Louis",
          "MU-RR" => "Rivière du Rempart",
          "MU-SA" => "Savanne"
        },
        "MV" => {
          "MV-01" => "Addu City",
          "MV-MLE" => "Male"
        },
        "MW" => {
          "MW-C" => "Central Region",
          "MW-N" => "Northern Region",
          "MW-S" => "Southern Region"
        },
        "MX" => {
          "MX-CMX" => "Ciudad de México",
          "MX-AGU" => "Aguascalientes",
          "MX-BCN" => "Baja California",
          "MX-BCS" => "Baja California Sur",
          "MX-CAM" => "Campeche",
          "MX-COA" => "Coahuila de Zaragoza",
          "MX-COL" => "Colima",
          "MX-CHP" => "Chiapas",
          "MX-CHH" => "Chihuahua",
          "MX-DUR" => "Durango",
          "MX-GUA" => "Guanajuato",
          "MX-GRO" => "Guerrero",
          "MX-HID" => "Hidalgo",
          "MX-JAL" => "Jalisco",
          "MX-MEX" => "México",
          "MX-MIC" => "Michoacán de Ocampo",
          "MX-MOR" => "Morelos",
          "MX-NAY" => "Nayarit",
          "MX-NLE" => "Nuevo León",
          "MX-OAX" => "Oaxaca",
          "MX-PUE" => "Puebla",
          "MX-QUE" => "Querétaro",
          "MX-ROO" => "Quintana Roo",
          "MX-SLP" => "San Luis Potosí",
          "MX-SIN" => "Sinaloa",
          "MX-SON" => "Sonora",
          "MX-TAB" => "Tabasco",
          "MX-TAM" => "Tamaulipas",
          "MX-TLA" => "Tlaxcala",
          "MX-VER" => "Veracruz de Ignacio de la Llave",
          "MX-YUC" => "Yucatán",
          "MX-ZAC" => "Zacatecas"
        },
        "MY" => {
          "MY-14" => "Wilayah Persekutuan Kuala Lumpur",
          "MY-15" => "Wilayah Persekutuan Labuan",
          "MY-16" => "Wilayah Persekutuan Putrajaya",
          "MY-01" => "Johor",
          "MY-02" => "Kedah",
          "MY-03" => "Kelantan",
          "MY-04" => "Melaka",
          "MY-05" => "Negeri Sembilan",
          "MY-06" => "Pahang",
          "MY-08" => "Perak",
          "MY-09" => "Perlis",
          "MY-07" => "Pulau Pinang",
          "MY-12" => "Sabah",
          "MY-13" => "Sarawak",
          "MY-10" => "Selangor",
          "MY-11" => "Terengganu"
        },
        "MZ" => {
          "MZ-MPM" => "Maputo",
          "MZ-P" => "Cabo Delgado",
          "MZ-G" => "Gaza",
          "MZ-I" => "Inhambane",
          "MZ-B" => "Manica",
          "MZ-L" => "Maputo",
          "MZ-N" => "Nampula",
          "MZ-A" => "Niassa",
          "MZ-S" => "Sofala",
          "MZ-T" => "Tete",
          "MZ-Q" => "Zambézia"
        },
        "NA" => {
          "NA-ER" => "Erongo",
          "NA-HA" => "Hardap",
          "NA-KA" => "//Karas(local variants are Karas and !Karas)",
          "NA-KE" => "Kavango East",
          "NA-KW" => "Kavango West",
          "NA-KH" => "Khomas",
          "NA-KU" => "Kunene",
          "NA-OW" => "Ohangwena",
          "NA-OH" => "Omaheke",
          "NA-OS" => "Omusati",
          "NA-ON" => "Oshana",
          "NA-OT" => "Oshikoto",
          "NA-OD" => "Otjozondjupa",
          "NA-CA" => "Zambezi"
        },
        "NE" => {
          "NE-8" => "Niamey",
          "NE-1" => "Agadez",
          "NE-2" => "Diffa",
          "NE-3" => "Dosso",
          "NE-4" => "Maradi",
          "NE-5" => "Tahoua",
          "NE-6" => "Tillabéri",
          "NE-7" => "Zinder"
        },
        "NG" => {
          "NG-FC" => "Abuja Federal Capital Territory",
          "NG-AB" => "Abia",
          "NG-AD" => "Adamawa",
          "NG-AK" => "Akwa Ibom",
          "NG-AN" => "Anambra",
          "NG-BA" => "Bauchi",
          "NG-BY" => "Bayelsa",
          "NG-BE" => "Benue",
          "NG-BO" => "Borno",
          "NG-CR" => "Cross River",
          "NG-DE" => "Delta",
          "NG-EB" => "Ebonyi",
          "NG-ED" => "Edo",
          "NG-EK" => "Ekiti",
          "NG-EN" => "Enugu",
          "NG-GO" => "Gombe",
          "NG-IM" => "Imo",
          "NG-JI" => "Jigawa",
          "NG-KD" => "Kaduna",
          "NG-KN" => "Kano",
          "NG-KT" => "Katsina",
          "NG-KE" => "Kebbi",
          "NG-KO" => "Kogi",
          "NG-KW" => "Kwara",
          "NG-LA" => "Lagos",
          "NG-NA" => "Nasarawa",
          "NG-NI" => "Niger",
          "NG-OG" => "Ogun",
          "NG-ON" => "Ondo",
          "NG-OS" => "Osun",
          "NG-OY" => "Oyo",
          "NG-PL" => "Plateau",
          "NG-RI" => "Rivers",
          "NG-SO" => "Sokoto",
          "NG-TA" => "Taraba",
          "NG-YO" => "Yobe",
          "NG-ZA" => "Zamfara"
        },
        "NI" => {
          "NI-BO" => "Boaco",
          "NI-CA" => "Carazo",
          "NI-CI" => "Chinandega",
          "NI-CO" => "Chontales",
          "NI-ES" => "Estelí",
          "NI-GR" => "Granada",
          "NI-JI" => "Jinotega",
          "NI-LE" => "León",
          "NI-MD" => "Madriz",
          "NI-MN" => "Managua",
          "NI-MS" => "Masaya",
          "NI-MT" => "Matagalpa",
          "NI-NS" => "Nueva Segovia",
          "NI-SJ" => "Río San Juan",
          "NI-RI" => "Rivas",
          "NI-AN" => "Costa Caribe Norte",
          "NI-AS" => "Costa Caribe Sur"
        },
        "NL" => {
          "NL-DR" => "Drenthe",
          "NL-FL" => "Flevoland",
          "NL-FR" => "Fryslân (fy)",
          "NL-GE" => "Gelderland",
          "NL-GR" => "Groningen",
          "NL-LI" => "Limburg",
          "NL-NB" => "Noord-Brabant",
          "NL-NH" => "Noord-Holland",
          "NL-OV" => "Overijssel",
          "NL-UT" => "Utrecht",
          "NL-ZE" => "Zeeland",
          "NL-ZH" => "Zuid-Holland"
        },
        "NO" => {
          "NO-42" => "Agder",
          "NO-34" => "Innlandet",
          "NO-15" => "Møre og Romsdal",
          "NO-18" => "Nordland",
          "NO-03" => "Oslo",
          "NO-11" => "Rogaland",
          "NO-54" => "Troms og Finnmark / Romsa ja Finnmárku (se) / Tromssan ja Finmarkun (-)",
          "NO-50" => "Trøndelag / Trööndelage (-)",
          "NO-38" => "Vestfold og Telemark",
          "NO-46" => "Vestland",
          "NO-30" => "Viken",
          "NO-22" => "Jan Mayen",
          "NO-21" => "Svalbard"
        },
        "NP" => {
          "NP-1" => "Madhyamanchal",
          "NP-2" => "Madhya Pashchimanchal",
          "NP-3" => "Pashchimanchal",
          "NP-4" => "Purwanchal",
          "NP-5" => "Sudur Pashchimanchal"
        },
        "NR" => {
          "NR-01" => "Aiwo",
          "NR-02" => "Anabar",
          "NR-03" => "Anetan",
          "NR-04" => "Anibare",
          "NR-05" => "Baitsi (local variant is Baiti)",
          "NR-06" => "Boe",
          "NR-07" => "Buada",
          "NR-08" => "Denigomodu",
          "NR-09" => "Ewa",
          "NR-10" => "Ijuw",
          "NR-11" => "Meneng",
          "NR-12" => "Nibok",
          "NR-13" => "Uaboe",
          "NR-14" => "Yaren"
        },
        "NZ" => {
          "NZ-AUK" => "Auckland",
          "NZ-BOP" => "Bay of Plenty",
          "NZ-CAN" => "Canterbury",
          "NZ-GIS" => "Gisborne",
          "NZ-HKB" => "Hawke's Bay",
          "NZ-MBH" => "Marlborough",
          "NZ-MWT" => "Manawatu-Wanganui",
          "NZ-NSN" => "Nelson",
          "NZ-NTL" => "Northland",
          "NZ-OTA" => "Otago",
          "NZ-STL" => "Southland",
          "NZ-TAS" => "Tasman",
          "NZ-TKI" => "Taranaki",
          "NZ-WGN" => "Wellington",
          "NZ-WTC" => "West Coast",
          "NZ-CIT" => "Chatham Islands Territory"
        },
        "OM" => {
          "OM-DA" => "Ad Dākhilīyah",
          "OM-BS" => "Shamāl al Bāţinah",
          "OM-BJ" => "Janūb al Bāţinah",
          "OM-WU" => "Al Wusţá",
          "OM-SS" => "Shamāl ash Sharqīyah",
          "OM-SJ" => "Janūb ash Sharqīyah",
          "OM-ZA" => "Az̧ Z̧āhirah",
          "OM-BU" => "Al Buraymī",
          "OM-MA" => "Masqaţ",
          "OM-MU" => "Musandam",
          "OM-ZU" => "Z̧ufār"
        },
        "PA" => {
          "PA-1" => "Bocas del Toro",
          "PA-4" => "Chiriquí",
          "PA-2" => "Coclé",
          "PA-3" => "Colón",
          "PA-5" => "Darién",
          "PA-EM" => "Emberá",
          "PA-KY" => "Guna Yala(local variant is Kuna Yala)",
          "PA-6" => "Herrera",
          "PA-7" => "Los Santos",
          "PA-NB" => "Ngöbe-Buglé[note 1]",
          "PA-8" => "Panamá",
          "PA-10" => "Panamá Oeste",
          "PA-9" => "Veraguas"
        },
        "PE" => {
          "PE-LMA" => "Municipalidad Metropolitana de Lima",
          "PE-AMA" => "Amazonas",
          "PE-ANC" => "Ancash",
          "PE-APU" => "Apurímac",
          "PE-ARE" => "Arequipa",
          "PE-AYA" => "Ayacucho",
          "PE-CAJ" => "Cajamarca",
          "PE-CUS" => "Cusco(local variant is Cuzco)",
          "PE-CAL" => "El Callao",
          "PE-HUV" => "Huancavelica",
          "PE-HUC" => "Huánuco",
          "PE-ICA" => "Ica",
          "PE-JUN" => "Junín",
          "PE-LAL" => "La Libertad",
          "PE-LAM" => "Lambayeque",
          "PE-LIM" => "Lima",
          "PE-LOR" => "Loreto",
          "PE-MDD" => "Madre de Dios",
          "PE-MOQ" => "Moquegua",
          "PE-PAS" => "Pasco",
          "PE-PIU" => "Piura",
          "PE-PUN" => "Puno",
          "PE-SAM" => "San Martín",
          "PE-TAC" => "Tacna",
          "PE-TUM" => "Tumbes",
          "PE-UCA" => "Ucayali"
        },
        "PG" => {
          "PG-NCD" => "National Capital District (Port Moresby)",
          "PG-CPM" => "Central",
          "PG-CPK" => "Chimbu",
          "PG-EHG" => "Eastern Highlands",
          "PG-EBR" => "East New Britain",
          "PG-ESW" => "East Sepik",
          "PG-EPW" => "Enga",
          "PG-GPK" => "Gulf",
          "PG-HLA" => "Hela",
          "PG-JWK" => "Jiwaka",
          "PG-MPM" => "Madang",
          "PG-MRL" => "Manus",
          "PG-MBA" => "Milne Bay",
          "PG-MPL" => "Morobe",
          "PG-NIK" => "New Ireland",
          "PG-NPP" => "Northern",
          "PG-SAN" => "West Sepik",
          "PG-SHM" => "Southern Highlands",
          "PG-WPD" => "Western",
          "PG-WHM" => "Western Highlands",
          "PG-WBK" => "West New Britain",
          "PG-NSB" => "Bougainville"
        },
        "PH" => {
          "PH-14" => "Autonomous Region in Muslim Mindanao[b]",
          "PH-05" => "Bicol",
          "PH-02" => "Cagayan Valley",
          "PH-40" => "Calabarzon",
          "PH-13" => "Caraga",
          "PH-03" => "Central Luzon",
          "PH-07" => "Central Visayas",
          "PH-15" => "Cordillera Administrative Region",
          "PH-11" => "Davao",
          "PH-08" => "Eastern Visayas",
          "PH-01" => "Ilocos",
          "PH-41" => "Mimaropa",
          "PH-00" => "National Capital Region",
          "PH-10" => "Northern Mindanao",
          "PH-12" => "Soccsksargen",
          "PH-06" => "Western Visayas",
          "PH-09" => "Zamboanga Peninsula"
        },
        "PK" => {
          "PK-IS" => "Islamabad",
          "PK-BA" => "Balochistan",
          "PK-KP" => "Khyber Pakhtunkhwa",
          "PK-PB" => "Punjab",
          "PK-SD" => "Sindh",
          "PK-JK" => "Azad Jammu and Kashmir(local variant is AJ&K)",
          "PK-GB" => "Gilgit-Baltistan"
        },
        "PL" => {
          "PL-02" => "Dolnośląskie",
          "PL-04" => "Kujawsko-pomorskie",
          "PL-06" => "Lubelskie",
          "PL-08" => "Lubuskie",
          "PL-10" => "Łódzkie",
          "PL-12" => "Małopolskie",
          "PL-14" => "Mazowieckie",
          "PL-16" => "Opolskie",
          "PL-18" => "Podkarpackie",
          "PL-20" => "Podlaskie",
          "PL-22" => "Pomorskie",
          "PL-24" => "Śląskie",
          "PL-26" => "Świętokrzyskie",
          "PL-28" => "Warmińsko-mazurskie",
          "PL-30" => "Wielkopolskie",
          "PL-32" => "Zachodniopomorskie"
        },
        "PS" => {
          "PS-BTH" => "Bethlehem",
          "PS-DEB" => "Deir El Balah",
          "PS-GZA" => "Gaza",
          "PS-HBN" => "Hebron",
          "PS-JEN" => "Jenin",
          "PS-JRH" => "Jericho and Al Aghwar",
          "PS-JEM" => "Jerusalem",
          "PS-KYS" => "Khan Yunis",
          "PS-NBS" => "Nablus",
          "PS-NGZ" => "North Gaza",
          "PS-QQA" => "Qalqilya",
          "PS-RFH" => "Rafah",
          "PS-RBH" => "Ramallah",
          "PS-SLT" => "Salfit",
          "PS-TBS" => "Tubas",
          "PS-TKM" => "Tulkarm"
        },
        "PT" => {
          "PT-01" => "Aveiro",
          "PT-02" => "Beja",
          "PT-03" => "Braga",
          "PT-04" => "Bragança",
          "PT-05" => "Castelo Branco",
          "PT-06" => "Coimbra",
          "PT-07" => "Évora",
          "PT-08" => "Faro",
          "PT-09" => "Guarda",
          "PT-10" => "Leiria",
          "PT-11" => "Lisboa",
          "PT-12" => "Portalegre",
          "PT-13" => "Porto",
          "PT-14" => "Santarém",
          "PT-15" => "Setúbal",
          "PT-16" => "Viana do Castelo",
          "PT-17" => "Vila Real",
          "PT-18" => "Viseu",
          "PT-20" => "Região Autónoma dos Açores",
          "PT-30" => "Região Autónoma da Madeira"
        },
        "PW" => {
          "PW-002" => "[[{{{alias}}}|Aimeliik]]",
          "PW-004" => "Airai",
          "PW-010" => "[[{{{alias}}}|Angaur]]",
          "PW-050" => "Hatohobei",
          "PW-100" => "Kayangel",
          "PW-150" => "Koror",
          "PW-212" => "Melekeok",
          "PW-214" => "Ngaraard",
          "PW-218" => "Ngarchelong",
          "PW-222" => "Ngardmau",
          "PW-224" => "Ngatpang",
          "PW-226" => "Ngchesar",
          "PW-227" => "Ngeremlengui",
          "PW-228" => "Ngiwal",
          "PW-350" => "Peleliu",
          "PW-370" => "Sonsorol"
        },
        "PY" => {
          "PY-ASU" => "Asunción",
          "PY-16" => "Alto Paraguay",
          "PY-10" => "Alto Paraná",
          "PY-13" => "Amambay",
          "PY-19" => "Boquerón",
          "PY-5" => "Caaguazú",
          "PY-6" => "Caazapá",
          "PY-14" => "Canindeyú",
          "PY-11" => "Central",
          "PY-1" => "Concepción",
          "PY-3" => "Cordillera",
          "PY-4" => "Guairá",
          "PY-7" => "Itapúa",
          "PY-8" => "Misiones",
          "PY-12" => "Ñeembucú",
          "PY-9" => "Paraguarí",
          "PY-15" => "Presidente Hayes",
          "PY-2" => "San Pedro"
        },
        "QA" => {
          "QA-DA" => "Ad Dawḩah",
          "QA-KH" => "Al Khawr wa adh Dhakhīrah",
          "QA-WA" => "Al Wakrah",
          "QA-RA" => "Ar Rayyān",
          "QA-MS" => "Ash Shamāl",
          "QA-SH" => "Ash Shīḩānīyah",
          "QA-ZA" => "Az̧ Z̧a‘āyin",
          "QA-US" => "Umm Şalāl"
        },
        "RO" => {
          "RO-AB" => "Alba",
          "RO-AR" => "Arad",
          "RO-AG" => "Argeș",
          "RO-BC" => "Bacău",
          "RO-BH" => "Bihor",
          "RO-BN" => "Bistrița-Năsăud",
          "RO-BT" => "Botoșani",
          "RO-BV" => "Brașov",
          "RO-BR" => "Brăila",
          "RO-BZ" => "Buzău",
          "RO-CS" => "Caraș-Severin",
          "RO-CL" => "Călărași",
          "RO-CJ" => "Cluj",
          "RO-CT" => "Constanța",
          "RO-CV" => "Covasna",
          "RO-DB" => "Dâmbovița",
          "RO-DJ" => "Dolj",
          "RO-GL" => "Galați",
          "RO-GR" => "Giurgiu",
          "RO-GJ" => "Gorj",
          "RO-HR" => "Harghita",
          "RO-HD" => "Hunedoara",
          "RO-IL" => "Ialomița",
          "RO-IS" => "Iași",
          "RO-IF" => "Ilfov",
          "RO-MM" => "Maramureș",
          "RO-MH" => "Mehedinți",
          "RO-MS" => "Mureș",
          "RO-NT" => "Neamț",
          "RO-OT" => "Olt",
          "RO-PH" => "Prahova",
          "RO-SM" => "Satu Mare",
          "RO-SJ" => "Sălaj",
          "RO-SB" => "Sibiu",
          "RO-SV" => "Suceava",
          "RO-TR" => "Teleorman",
          "RO-TM" => "Timiș",
          "RO-TL" => "Tulcea",
          "RO-VS" => "Vaslui",
          "RO-VL" => "Vâlcea",
          "RO-VN" => "Vrancea",
          "RO-B" => "București"
        },
        "RS" => {
          "RS-KM" => "Kosovo-Metohija[a]",
          "RS-VO" => "Vojvodina"
        },
        "RU" => {
          "RU-AD" => "Adygeya, Respublika",
          "RU-AL" => "Altay, Respublika",
          "RU-BA" => "Bashkortostan, Respublika",
          "RU-BU" => "Buryatiya, Respublika",
          "RU-CE" => "Chechenskaya Respublika",
          "RU-CU" => "Chuvashskaya Respublika",
          "RU-DA" => "Dagestan, Respublika",
          "RU-IN" => "Ingushetiya, Respublika",
          "RU-KB" => "Kabardino-Balkarskaya Respublika",
          "RU-KL" => "Kalmykiya, Respublika",
          "RU-KC" => "Karachayevo-Cherkesskaya Respublika",
          "RU-KR" => "Kareliya, Respublika",
          "RU-KK" => "Khakasiya, Respublika",
          "RU-KO" => "Komi, Respublika",
          "RU-ME" => "Mariy El, Respublika",
          "RU-MO" => "Mordoviya, Respublika",
          "RU-SA" => "Saha, Respublika(local variant is Jakutija)",
          "RU-SE" => "Severnaya Osetiya, Respublika(local variant is Alaniya [Respublika Severnaya Osetiya – Alaniya])",
          "RU-TA" => "Tatarstan, Respublika",
          "RU-TY" => "Tyva, Respublika(local variant is Tuva)",
          "RU-UD" => "Udmurtskaya Respublika",
          "RU-ALT" => "Altayskiy kray",
          "RU-KAM" => "Kamchatskiy kray",
          "RU-KHA" => "Khabarovskiy kray",
          "RU-KDA" => "Krasnodarskiy kray",
          "RU-KYA" => "Krasnoyarskiy kray",
          "RU-PER" => "Permskiy kray",
          "RU-PRI" => "Primorskiy kray",
          "RU-STA" => "Stavropol'skiy kray",
          "RU-ZAB" => "Zabaykal'skiy kray",
          "RU-AMU" => "Amurskaya oblast'",
          "RU-ARK" => "Arkhangel'skaya oblast'",
          "RU-AST" => "Astrakhanskaya oblast'",
          "RU-BEL" => "Belgorodskaya oblast'",
          "RU-BRY" => "Bryanskaya oblast'",
          "RU-CHE" => "Chelyabinskaya oblast'",
          "RU-IRK" => "Irkutskaya oblast'",
          "RU-IVA" => "Ivanovskaya oblast'",
          "RU-KGD" => "Kaliningradskaya oblast'",
          "RU-KLU" => "Kaluzhskaya oblast'",
          "RU-KEM" => "Kemerovskaya oblast'",
          "RU-KIR" => "Kirovskaya oblast'",
          "RU-KOS" => "Kostromskaya oblast'",
          "RU-KGN" => "Kurganskaya oblast'",
          "RU-KRS" => "Kurskaya oblast'",
          "RU-LEN" => "Leningradskaya oblast'",
          "RU-LIP" => "Lipetskaya oblast'",
          "RU-MAG" => "Magadanskaya oblast'",
          "RU-MOS" => "Moskovskaya oblast'",
          "RU-MUR" => "Murmanskaya oblast'",
          "RU-NIZ" => "Nizhegorodskaya oblast'",
          "RU-NGR" => "Novgorodskaya oblast'",
          "RU-NVS" => "Novosibirskaya oblast'",
          "RU-OMS" => "Omskaya oblast'",
          "RU-ORE" => "Orenburgskaya oblast'",
          "RU-ORL" => "Orlovskaya oblast'",
          "RU-PNZ" => "Penzenskaya oblast'",
          "RU-PSK" => "Pskovskaya oblast'",
          "RU-ROS" => "Rostovskaya oblast'",
          "RU-RYA" => "Ryazanskaya oblast'",
          "RU-SAK" => "Sakhalinskaya oblast'",
          "RU-SAM" => "Samarskaya oblast'",
          "RU-SAR" => "Saratovskaya oblast'",
          "RU-SMO" => "Smolenskaya oblast'",
          "RU-SVE" => "Sverdlovskaya oblast'",
          "RU-TAM" => "Tambovskaya oblast'",
          "RU-TOM" => "Tomskaya oblast'",
          "RU-TUL" => "Tul'skaya oblast'",
          "RU-TVE" => "Tverskaya oblast'",
          "RU-TYU" => "Tyumenskaya oblast'",
          "RU-ULY" => "Ul'yanovskaya oblast'",
          "RU-VLA" => "Vladimirskaya oblast'",
          "RU-VGG" => "Volgogradskaya oblast'",
          "RU-VLG" => "Vologodskaya oblast'",
          "RU-VOR" => "Voronezhskaya oblast'",
          "RU-YAR" => "Yaroslavskaya oblast'",
          "RU-MOW" => "Moskva",
          "RU-SPE" => "Sankt-Peterburg",
          "RU-YEV" => "Yevreyskaya avtonomnaya oblast'",
          "RU-CHU" => "Chukotskiy avtonomnyy okrug",
          "RU-KHM" => "Khanty-Mansiyskiy avtonomnyy okrug(local variant is Yugra)",
          "RU-NEN" => "Nenetskiy avtonomnyy okrug",
          "RU-YAN" => "Yamalo-Nenetskiy avtonomnyy okrug"
        },
        "RW" => {
          "RW-01" => "City of Kigali",
          "RW-02" => "Eastern",
          "RW-03" => "Northern",
          "RW-04" => "Western",
          "RW-05" => "Southern"
        },
        "SA" => {
          "SA-11" => "Al Bāḩah",
          "SA-08" => "Al Ḩudūd ash Shamālīyah",
          "SA-12" => "Al Jawf",
          "SA-03" => "Al Madīnah al Munawwarah",
          "SA-05" => "Al Qaşīm",
          "SA-01" => "Ar Riyāḑ",
          "SA-04" => "Ash Sharqīyah",
          "SA-14" => "'Asīr",
          "SA-06" => "Ḩā'il",
          "SA-09" => "Jāzān",
          "SA-02" => "Makkah al Mukarramah",
          "SA-10" => "Najrān",
          "SA-07" => "Tabūk"
        },
        "SB" => {
          "SB-CT" => "Capital Territory (Honiara)",
          "SB-CE" => "Central",
          "SB-CH" => "Choiseul",
          "SB-GU" => "Guadalcanal",
          "SB-IS" => "Isabel",
          "SB-MK" => "Makira-Ulawa",
          "SB-ML" => "Malaita",
          "SB-RB" => "Rennell and Bellona",
          "SB-TE" => "Temotu",
          "SB-WE" => "Western"
        },
        "SC" => {
          "SC-01" => "Anse aux Pins",
          "SC-02" => "Anse Boileau",
          "SC-03" => "Anse Etoile",
          "SC-05" => "Anse Royale",
          "SC-04" => "Au Cap",
          "SC-06" => "Baie Lazare",
          "SC-07" => "Baie Sainte Anne",
          "SC-08" => "Beau Vallon",
          "SC-09" => "Bel Air",
          "SC-10" => "Bel Ombre",
          "SC-11" => "Cascade",
          "SC-12" => "Glacis",
          "SC-13" => "Grand Anse Mahe",
          "SC-14" => "Grand Anse Praslin",
          "SC-15" => "La Digue",
          "SC-16" => "English River",
          "SC-26" => "Ile Perseverance I",
          "SC-27" => "Ile Perseverance II",
          "SC-24" => "Les Mamelles",
          "SC-17" => "Mont Buxton",
          "SC-18" => "Mont Fleuri",
          "SC-19" => "Plaisance",
          "SC-20" => "Pointe Larue",
          "SC-21" => "Port Glaud",
          "SC-25" => "Roche Caiman",
          "SC-22" => "Saint Louis",
          "SC-23" => "Takamaka"
        },
        "SD" => {
          "SD-RS" => "Al Baḩr al Aḩmar",
          "SD-GZ" => "Al Jazīrah",
          "SD-KH" => "Al Kharţūm",
          "SD-GD" => "Al Qaḑārif",
          "SD-NR" => "Nahr an Nīl",
          "SD-NW" => "An Nīl al Abyaḑ",
          "SD-NB" => "An Nīl al Azraq",
          "SD-NO" => "Ash Shamālīyah",
          "SD-DW" => "Gharb Dārfūr",
          "SD-GK" => "Gharb Kurdufān",
          "SD-DS" => "Janūb Dārfūr",
          "SD-KS" => "Janūb Kurdufān",
          "SD-KA" => "Kassalā",
          "SD-DN" => "Shamāl Dārfūr",
          "SD-KN" => "Shamāl Kurdufān",
          "SD-DE" => "Sharq Dārfūr",
          "SD-SI" => "Sinnār",
          "SD-DC" => "Wasaţ Dārfūr(local variant is Zālinjay)"
        },
        "SE" => {
          "SE-K" => "Blekinge län",
          "SE-W" => "Dalarnas län",
          "SE-I" => "Gotlands län",
          "SE-X" => "Gävleborgs län",
          "SE-N" => "Hallands län",
          "SE-Z" => "Jämtlands län",
          "SE-F" => "Jönköpings län",
          "SE-H" => "Kalmar län",
          "SE-G" => "Kronobergs län",
          "SE-BD" => "Norrbottens län",
          "SE-M" => "Skåne län",
          "SE-AB" => "Stockholms län",
          "SE-D" => "Södermanlands län",
          "SE-C" => "Uppsala län",
          "SE-S" => "Värmlands län",
          "SE-AC" => "Västerbottens län",
          "SE-Y" => "Västernorrlands län",
          "SE-U" => "Västmanlands län",
          "SE-O" => "Västra Götalands län",
          "SE-T" => "Örebro län",
          "SE-E" => "Östergötlands län"
        },
        "SG" => {
          "SG-01" => "Central Singapore",
          "SG-02" => "North East",
          "SG-03" => "North West",
          "SG-04" => "South East",
          "SG-05" => "South West"
        },
        "SH" => {
          "SH-AC" => "Ascension",
          "SH-HL" => "Saint Helena",
          "SH-TA" => "Tristan da Cunha"
        },
        "SI" => {
          "SI-001" => "Ajdovščina",
          "SI-213" => "Ankaran",
          "SI-195" => "Apače",
          "SI-002" => "Beltinci",
          "SI-148" => "Benedikt",
          "SI-149" => "Bistrica ob Sotli",
          "SI-003" => "Bled",
          "SI-150" => "Bloke",
          "SI-004" => "Bohinj",
          "SI-005" => "Borovnica",
          "SI-006" => "Bovec",
          "SI-151" => "Braslovče",
          "SI-007" => "Brda",
          "SI-008" => "Brezovica",
          "SI-009" => "Brežice",
          "SI-152" => "Cankova",
          "SI-011" => "Celje",
          "SI-012" => "Cerklje na Gorenjskem",
          "SI-013" => "Cerknica",
          "SI-014" => "Cerkno",
          "SI-153" => "Cerkvenjak",
          "SI-196" => "Cirkulane",
          "SI-015" => "Črenšovci",
          "SI-016" => "Črna na Koroškem",
          "SI-017" => "Črnomelj",
          "SI-018" => "Destrnik",
          "SI-019" => "Divača",
          "SI-154" => "Dobje",
          "SI-020" => "Dobrepolje",
          "SI-155" => "Dobrna",
          "SI-021" => "Dobrova-Polhov Gradec",
          "SI-156" => "Dobrovnik",
          "SI-022" => "Dol pri Ljubljani",
          "SI-157" => "Dolenjske Toplice",
          "SI-023" => "Domžale",
          "SI-024" => "Dornava",
          "SI-025" => "Dravograd",
          "SI-026" => "Duplek",
          "SI-027" => "Gorenja vas-Poljane",
          "SI-028" => "Gorišnica",
          "SI-207" => "Gorje",
          "SI-029" => "Gornja Radgona",
          "SI-030" => "Gornji Grad",
          "SI-031" => "Gornji Petrovci",
          "SI-158" => "Grad",
          "SI-032" => "Grosuplje",
          "SI-159" => "Hajdina",
          "SI-160" => "Hoče-Slivnica",
          "SI-161" => "Hodoš",
          "SI-162" => "Horjul",
          "SI-034" => "Hrastnik",
          "SI-035" => "Hrpelje-Kozina",
          "SI-036" => "Idrija",
          "SI-037" => "Ig",
          "SI-038" => "Ilirska Bistrica",
          "SI-039" => "Ivančna Gorica",
          "SI-040" => "Izola",
          "SI-041" => "Jesenice",
          "SI-163" => "Jezersko",
          "SI-042" => "Juršinci",
          "SI-043" => "Kamnik",
          "SI-044" => "Kanal",
          "SI-045" => "Kidričevo",
          "SI-046" => "Kobarid",
          "SI-047" => "Kobilje",
          "SI-048" => "Kočevje",
          "SI-049" => "Komen",
          "SI-164" => "Komenda",
          "SI-050" => "Koper",
          "SI-197" => "Kosanjevica na Krki",
          "SI-165" => "Kostel",
          "SI-051" => "Kozje",
          "SI-052" => "Kranj",
          "SI-053" => "Kranjska Gora",
          "SI-166" => "Križevci",
          "SI-054" => "Krško",
          "SI-055" => "Kungota",
          "SI-056" => "Kuzma",
          "SI-057" => "Laško",
          "SI-058" => "Lenart",
          "SI-059" => "Lendava",
          "SI-060" => "Litija",
          "SI-061" => "Ljubljana",
          "SI-062" => "Ljubno",
          "SI-063" => "Ljutomer",
          "SI-208" => "Log-Dragomer",
          "SI-064" => "Logatec",
          "SI-065" => "Loška dolina",
          "SI-066" => "Loški Potok",
          "SI-167" => "Lovrenc na Pohorju",
          "SI-067" => "Luče",
          "SI-068" => "Lukovica",
          "SI-069" => "Majšperk",
          "SI-198" => "Makole",
          "SI-070" => "Maribor",
          "SI-168" => "Markovci",
          "SI-071" => "Medvode",
          "SI-072" => "Mengeš",
          "SI-073" => "Metlika",
          "SI-074" => "Mežica",
          "SI-169" => "Miklavž na Dravskem polju",
          "SI-075" => "Miren-Kostanjevica",
          "SI-212" => "Mirna",
          "SI-170" => "Mirna Peč",
          "SI-076" => "Mislinja",
          "SI-199" => "Mokronog-Trebelno",
          "SI-077" => "Moravče",
          "SI-078" => "Moravske Toplice",
          "SI-079" => "Mozirje",
          "SI-080" => "Murska Sobota",
          "SI-081" => "Muta",
          "SI-082" => "Naklo",
          "SI-083" => "Nazarje",
          "SI-084" => "Nova Gorica",
          "SI-085" => "Novo Mesto",
          "SI-086" => "Odranci",
          "SI-171" => "Oplotnica",
          "SI-087" => "Ormož",
          "SI-088" => "Osilnica",
          "SI-089" => "Pesnica",
          "SI-090" => "Piran",
          "SI-091" => "Pivka",
          "SI-092" => "Podčetrtek",
          "SI-172" => "Podlehnik",
          "SI-093" => "Podvelka",
          "SI-200" => "Poljčane",
          "SI-173" => "Polzela",
          "SI-094" => "Postojna",
          "SI-174" => "Prebold",
          "SI-095" => "Preddvor",
          "SI-175" => "Prevalje",
          "SI-096" => "Ptuj",
          "SI-097" => "Puconci",
          "SI-098" => "Rače-Fram",
          "SI-099" => "Radeče",
          "SI-100" => "Radenci",
          "SI-101" => "Radlje ob Dravi",
          "SI-102" => "Radovljica",
          "SI-103" => "Ravne na Koroškem",
          "SI-176" => "Razkrižje",
          "SI-209" => "Rečica ob Savinji",
          "SI-201" => "Renče-Vogrsko",
          "SI-104" => "Ribnica",
          "SI-177" => "Ribnica na Pohorju",
          "SI-106" => "Rogaška Slatina",
          "SI-105" => "Rogašovci",
          "SI-107" => "Rogatec",
          "SI-108" => "Ruše",
          "SI-178" => "Selnica ob Dravi",
          "SI-109" => "Semič",
          "SI-110" => "Sevnica",
          "SI-111" => "Sežana",
          "SI-112" => "Slovenj Gradec",
          "SI-113" => "Slovenska Bistrica",
          "SI-114" => "Slovenske Konjice",
          "SI-179" => "Sodražica",
          "SI-180" => "Solčava",
          "SI-202" => "Središče ob Dravi",
          "SI-115" => "Starše",
          "SI-203" => "Straža",
          "SI-181" => "Sveta Ana",
          "SI-204" => "Sveta Trojica v Slovenskih goricah",
          "SI-182" => "Sveti Andraž v Slovenskih goricah",
          "SI-116" => "Sveti Jurij ob Ščavnici",
          "SI-210" => "Sveti Jurij v Slovenskih goricah",
          "SI-205" => "Sveti Tomaž",
          "SI-033" => "Šalovci",
          "SI-183" => "Šempeter-Vrtojba",
          "SI-117" => "Šenčur",
          "SI-118" => "Šentilj",
          "SI-119" => "Šentjernej",
          "SI-120" => "Šentjur",
          "SI-211" => "Šentrupert",
          "SI-121" => "Škocjan",
          "SI-122" => "Škofja Loka",
          "SI-123" => "Škofljica",
          "SI-124" => "Šmarje pri Jelšah",
          "SI-206" => "Šmarješke Toplice",
          "SI-125" => "Šmartno ob Paki",
          "SI-194" => "Šmartno pri Litiji",
          "SI-126" => "Šoštanj",
          "SI-127" => "Štore",
          "SI-184" => "Tabor",
          "SI-010" => "Tišina",
          "SI-128" => "Tolmin",
          "SI-129" => "Trbovlje",
          "SI-130" => "Trebnje",
          "SI-185" => "Trnovska Vas",
          "SI-186" => "Trzin",
          "SI-131" => "Tržič",
          "SI-132" => "Turnišče",
          "SI-133" => "Velenje",
          "SI-187" => "Velika Polana",
          "SI-134" => "Velike Lašče",
          "SI-188" => "Veržej",
          "SI-135" => "Videm",
          "SI-136" => "Vipava",
          "SI-137" => "Vitanje",
          "SI-138" => "Vodice",
          "SI-139" => "Vojnik",
          "SI-189" => "Vransko",
          "SI-140" => "Vrhnika",
          "SI-141" => "Vuzenica",
          "SI-142" => "Zagorje ob Savi",
          "SI-143" => "Zavrč",
          "SI-144" => "Zreče",
          "SI-190" => "Žalec",
          "SI-146" => "Železniki",
          "SI-191" => "Žetale",
          "SI-147" => "Žiri",
          "SI-192" => "Žirovnica",
          "SI-193" => "Žužemberk"
        },
        "SK" => {
          "SK-BC" => "Banskobystrický kraj",
          "SK-BL" => "Bratislavský kraj",
          "SK-KI" => "Košický kraj",
          "SK-NI" => "Nitriansky kraj",
          "SK-PV" => "Prešovský kraj",
          "SK-TC" => "Trenčiansky kraj",
          "SK-TA" => "Trnavský kraj",
          "SK-ZI" => "Žilinský kraj"
        },
        "SL" => {
          "SL-W" => "Western Area (Freetown)",
          "SL-E" => "Eastern",
          "SL-N" => "Northern",
          "SL-NW" => "North Western",
          "SL-S" => "Southern"
        },
        "SM" => {
          "SM-01" => "Acquaviva",
          "SM-06" => "Borgo Maggiore",
          "SM-02" => "Chiesanuova",
          "SM-07" => "Città di San Marino(local variant is San Marino)",
          "SM-03" => "Domagnano",
          "SM-04" => "Faetano",
          "SM-05" => "Fiorentino",
          "SM-08" => "Montegiardino",
          "SM-09" => "Serravalle"
        },
        "SN" => {
          "SN-DK" => "Dakar",
          "SN-DB" => "Diourbel",
          "SN-FK" => "Fatick",
          "SN-KA" => "Kaffrine",
          "SN-KL" => "Kaolack",
          "SN-KE" => "Kédougou",
          "SN-KD" => "Kolda",
          "SN-LG" => "Louga",
          "SN-MT" => "Matam",
          "SN-SL" => "Saint-Louis",
          "SN-SE" => "Sédhiou",
          "SN-TC" => "Tambacounda",
          "SN-TH" => "Thiès",
          "SN-ZG" => "Ziguinchor"
        },
        "SO" => {
          "SO-AW" => "Awdal",
          "SO-BK" => "Bakool",
          "SO-BN" => "Banaadir",
          "SO-BR" => "Bari",
          "SO-BY" => "Bay",
          "SO-GA" => "Galguduud",
          "SO-GE" => "Gedo",
          "SO-HI" => "Hiiraan",
          "SO-JD" => "Jubbada Dhexe",
          "SO-JH" => "Jubbada Hoose",
          "SO-MU" => "Mudug",
          "SO-NU" => "Nugaal",
          "SO-SA" => "Sanaag",
          "SO-SD" => "Shabeellaha Dhexe",
          "SO-SH" => "Shabeellaha Hoose",
          "SO-SO" => "Sool",
          "SO-TO" => "Togdheer",
          "SO-WO" => "Woqooyi Galbeed"
        },
        "SR" => {
          "SR-BR" => "Brokopondo",
          "SR-CM" => "Commewijne",
          "SR-CR" => "Coronie",
          "SR-MA" => "Marowijne",
          "SR-NI" => "Nickerie",
          "SR-PR" => "Para",
          "SR-PM" => "Paramaribo",
          "SR-SA" => "Saramacca",
          "SR-SI" => "Sipaliwini",
          "SR-WA" => "Wanica"
        },
        "SS" => {
          "SS-EC" => "Central Equatoria",
          "SS-EE" => "Eastern Equatoria",
          "SS-JG" => "Jonglei",
          "SS-LK" => "Lakes",
          "SS-BN" => "Northern Bahr el Ghazal",
          "SS-UY" => "Unity",
          "SS-NU" => "Upper Nile",
          "SS-WR" => "Warrap",
          "SS-BW" => "Western Bahr el Ghazal",
          "SS-EW" => "Western Equatoria"
        },
        "ST" => {
          "ST-01" => "Água Grande",
          "ST-02" => "Cantagalo",
          "ST-03" => "Caué",
          "ST-04" => "Lembá",
          "ST-05" => "Lobata",
          "ST-06" => "Mé-Zóchi",
          "ST-P" => "Príncipe"
        },
        "SV" => {
          "SV-AH" => "Ahuachapán",
          "SV-CA" => "Cabañas",
          "SV-CH" => "Chalatenango",
          "SV-CU" => "Cuscatlán",
          "SV-LI" => "La Libertad",
          "SV-PA" => "La Paz",
          "SV-UN" => "La Unión",
          "SV-MO" => "Morazán",
          "SV-SM" => "San Miguel",
          "SV-SS" => "San Salvador",
          "SV-SV" => "San Vicente",
          "SV-SA" => "Santa Ana",
          "SV-SO" => "Sonsonate",
          "SV-US" => "Usulután"
        },
        "SY" => {
          "SY-HA" => "Al Ḩasakah",
          "SY-LA" => "Al Lādhiqīyah",
          "SY-QU" => "Al Qunayţirah",
          "SY-RA" => "Ar Raqqah",
          "SY-SU" => "As Suwaydā'",
          "SY-DR" => "Dar'ā",
          "SY-DY" => "Dayr az Zawr",
          "SY-DI" => "Dimashq",
          "SY-HL" => "Ḩalab",
          "SY-HM" => "Ḩamāh",
          "SY-HI" => "Ḩimş",
          "SY-ID" => "Idlib",
          "SY-RD" => "Rīf Dimashq",
          "SY-TA" => "Ţarţūs"
        },
        "SZ" => {
          "SZ-HH" => "Hhohho",
          "SZ-LU" => "Lubombo",
          "SZ-MA" => "Manzini",
          "SZ-SH" => "Shiselweni"
        },
        "TD" => {
          "TD-BA" => "Al Baţḩā’",
          "TD-LC" => "Al Buḩayrah",
          "TD-BG" => "Baḩr al Ghazāl",
          "TD-BO" => "Būrkū",
          "TD-HL" => "Ḩajjar Lamīs",
          "TD-EO" => "Inīdī al Gharbī",
          "TD-EE" => "Inīdī ash Sharqī",
          "TD-KA" => "Kānim",
          "TD-LO" => "Lūghūn al Gharbī",
          "TD-LR" => "Lūghūn ash Sharqī",
          "TD-ND" => "Madīnat Injamīnā",
          "TD-MA" => "Māndūl",
          "TD-MO" => "Māyū Kībbī al Gharbī",
          "TD-ME" => "Māyū Kībbī ash Sharqī",
          "TD-GR" => "Qīrā",
          "TD-SA" => "Salāmāt",
          "TD-MC" => "Shārī al Awsaţ",
          "TD-CB" => "Shārī Bāqirmī",
          "TD-SI" => "Sīlā",
          "TD-TA" => "Tānjīlī",
          "TD-TI" => "Tibastī",
          "TD-OD" => "Waddāy",
          "TD-WF" => "Wādī Fīrā’"
        },
        "TG" => {
          "TG-C" => "Centrale",
          "TG-K" => "Kara",
          "TG-M" => "Maritime (Région)",
          "TG-P" => "Plateaux",
          "TG-S" => "Savanes"
        },
        "TH" => {
          "TH-10" => "Krung Thep Maha Nakhon(local variant is Bangkok)",
          "TH-S" => "Phatthaya",
          "TH-37" => "Amnat Charoen",
          "TH-15" => "Ang Thong",
          "TH-38" => "Bueng Kan",
          "TH-31" => "Buri Ram",
          "TH-24" => "Chachoengsao",
          "TH-18" => "Chai Nat",
          "TH-36" => "Chaiyaphum",
          "TH-22" => "Chanthaburi",
          "TH-50" => "Chiang Mai",
          "TH-57" => "Chiang Rai",
          "TH-20" => "Chon Buri",
          "TH-86" => "Chumphon",
          "TH-46" => "Kalasin",
          "TH-62" => "Kamphaeng Phet",
          "TH-71" => "Kanchanaburi",
          "TH-40" => "Khon Kaen",
          "TH-81" => "Krabi",
          "TH-52" => "Lampang",
          "TH-51" => "Lamphun",
          "TH-42" => "Loei",
          "TH-16" => "Lop Buri",
          "TH-58" => "Mae Hong Son",
          "TH-44" => "Maha Sarakham",
          "TH-49" => "Mukdahan",
          "TH-26" => "Nakhon Nayok",
          "TH-73" => "Nakhon Pathom",
          "TH-48" => "Nakhon Phanom",
          "TH-30" => "Nakhon Ratchasima",
          "TH-60" => "Nakhon Sawan",
          "TH-80" => "Nakhon Si Thammarat",
          "TH-55" => "Nan",
          "TH-96" => "Narathiwat",
          "TH-39" => "Nong Bua Lam Phu",
          "TH-43" => "Nong Khai",
          "TH-12" => "Nonthaburi",
          "TH-13" => "Pathum Thani",
          "TH-94" => "Pattani",
          "TH-82" => "Phangnga",
          "TH-93" => "Phatthalung",
          "TH-56" => "Phayao",
          "TH-67" => "Phetchabun",
          "TH-76" => "Phetchaburi",
          "TH-66" => "Phichit",
          "TH-65" => "Phitsanulok",
          "TH-54" => "Phrae",
          "TH-14" => "Phra Nakhon Si Ayutthaya",
          "TH-83" => "Phuket",
          "TH-25" => "Prachin Buri",
          "TH-77" => "Prachuap Khiri Khan",
          "TH-85" => "Ranong",
          "TH-70" => "Ratchaburi",
          "TH-21" => "Rayong",
          "TH-45" => "Roi Et",
          "TH-27" => "Sa Kaeo",
          "TH-47" => "Sakon Nakhon",
          "TH-11" => "Samut Prakan",
          "TH-74" => "Samut Sakhon",
          "TH-75" => "Samut Songkhram",
          "TH-19" => "Saraburi",
          "TH-91" => "Satun",
          "TH-17" => "Sing Buri",
          "TH-33" => "Si Sa Ket",
          "TH-90" => "Songkhla",
          "TH-64" => "Sukhothai",
          "TH-72" => "Suphan Buri",
          "TH-84" => "Surat Thani",
          "TH-32" => "Surin",
          "TH-63" => "Tak",
          "TH-92" => "Trang",
          "TH-23" => "Trat",
          "TH-34" => "Ubon Ratchathani",
          "TH-41" => "Udon Thani",
          "TH-61" => "Uthai Thani",
          "TH-53" => "Uttaradit",
          "TH-95" => "Yala",
          "TH-35" => "Yasothon"
        },
        "TJ" => {
          "TJ-DU" => "Dushanbe",
          "TJ-GB" => "Kŭhistoni Badakhshon",
          "TJ-KT" => "Khatlon",
          "TJ-SU" => "Sughd",
          "TJ-RA" => "nohiyahoi tobei jumhurí"
        },
        "TL" => {
          "TL-AL" => "Aileu",
          "TL-AN" => "Ainaro",
          "TL-BA" => "Baucau",
          "TL-BO" => "Bobonaro",
          "TL-CO" => "Cova Lima",
          "TL-DI" => "Díli",
          "TL-ER" => "Ermera",
          "TL-LA" => "Lautém",
          "TL-LI" => "Liquiça",
          "TL-MT" => "Manatuto",
          "TL-MF" => "Manufahi",
          "TL-OE" => "Oé-Cusse Ambeno(local variant is Oecussi)",
          "TL-VI" => "Viqueque"
        },
        "TM" => {
          "TM-A" => "Ahal",
          "TM-B" => "Balkan",
          "TM-D" => "Daşoguz",
          "TM-L" => "Lebap",
          "TM-M" => "Mary",
          "TM-S" => "Aşgabat"
        },
        "TN" => {
          "TN-31" => "Béja",
          "TN-13" => "Ben Arous",
          "TN-23" => "Bizerte",
          "TN-81" => "Gabès",
          "TN-71" => "Gafsa",
          "TN-32" => "Jendouba",
          "TN-41" => "Kairouan",
          "TN-42" => "Kasserine",
          "TN-73" => "Kébili",
          "TN-12" => "L'Ariana",
          "TN-14" => "La Manouba",
          "TN-33" => "Le Kef",
          "TN-53" => "Mahdia",
          "TN-82" => "Médenine",
          "TN-52" => "Monastir",
          "TN-21" => "Nabeul",
          "TN-61" => "Sfax",
          "TN-43" => "Sidi Bouzid",
          "TN-34" => "Siliana",
          "TN-51" => "Sousse",
          "TN-83" => "Tataouine",
          "TN-72" => "Tozeur",
          "TN-11" => "Tunis",
          "TN-22" => "Zaghouan"
        },
        "TO" => {
          "TO-01" => "'Eua",
          "TO-02" => "Ha'apai",
          "TO-03" => "Niuas",
          "TO-04" => "Tongatapu",
          "TO-05" => "Vava'u"
        },
        "TR" => {
          "TR-01" => "Adana",
          "TR-02" => "Adıyaman",
          "TR-03" => "Afyonkarahisar",
          "TR-04" => "Ağrı",
          "TR-68" => "Aksaray",
          "TR-05" => "Amasya",
          "TR-06" => "Ankara",
          "TR-07" => "Antalya",
          "TR-75" => "Ardahan",
          "TR-08" => "Artvin",
          "TR-09" => "Aydın",
          "TR-10" => "Balıkesir",
          "TR-74" => "Bartın",
          "TR-72" => "Batman",
          "TR-69" => "Bayburt",
          "TR-11" => "Bilecik",
          "TR-12" => "Bingöl",
          "TR-13" => "Bitlis",
          "TR-14" => "Bolu",
          "TR-15" => "Burdur",
          "TR-16" => "Bursa",
          "TR-17" => "Çanakkale",
          "TR-18" => "Çankırı",
          "TR-19" => "Çorum",
          "TR-20" => "Denizli",
          "TR-21" => "Diyarbakır",
          "TR-81" => "Düzce",
          "TR-22" => "Edirne",
          "TR-23" => "Elazığ",
          "TR-24" => "Erzincan",
          "TR-25" => "Erzurum",
          "TR-26" => "Eskişehir",
          "TR-27" => "Gaziantep",
          "TR-28" => "Giresun",
          "TR-29" => "Gümüşhane",
          "TR-30" => "Hakkâri",
          "TR-31" => "Hatay",
          "TR-76" => "Iğdır",
          "TR-32" => "Isparta",
          "TR-34" => "İstanbul",
          "TR-35" => "İzmir",
          "TR-46" => "Kahramanmaraş",
          "TR-78" => "Karabük",
          "TR-70" => "Karaman",
          "TR-36" => "Kars",
          "TR-37" => "Kastamonu",
          "TR-38" => "Kayseri",
          "TR-71" => "Kırıkkale",
          "TR-39" => "Kırklareli",
          "TR-40" => "Kırşehir",
          "TR-79" => "Kilis",
          "TR-41" => "Kocaeli",
          "TR-42" => "Konya",
          "TR-43" => "Kütahya",
          "TR-44" => "Malatya",
          "TR-45" => "Manisa",
          "TR-47" => "Mardin",
          "TR-33" => "Mersin",
          "TR-48" => "Muğla",
          "TR-49" => "Muş",
          "TR-50" => "Nevşehir",
          "TR-51" => "Niğde",
          "TR-52" => "Ordu",
          "TR-80" => "Osmaniye",
          "TR-53" => "Rize",
          "TR-54" => "Sakarya",
          "TR-55" => "Samsun",
          "TR-56" => "Siirt",
          "TR-57" => "Sinop",
          "TR-58" => "Sivas",
          "TR-63" => "Şanlıurfa",
          "TR-73" => "Şırnak",
          "TR-59" => "Tekirdağ",
          "TR-60" => "Tokat",
          "TR-61" => "Trabzon",
          "TR-62" => "Tunceli",
          "TR-64" => "Uşak",
          "TR-65" => "Van",
          "TR-77" => "Yalova",
          "TR-66" => "Yozgat",
          "TR-67" => "Zonguldak"
        },
        "TT" => {
          "TT-ARI" => "Arima",
          "TT-CHA" => "Chaguanas",
          "TT-CTT" => "Couva–Tabaquite–Talparo",
          "TT-DMN" => "Diego Martin",
          "TT-MRC" => "Mayaro-Rio Claro",
          "TT-PED" => "Penal-Debe",
          "TT-POS" => "Port of Spain",
          "TT-PRT" => "Princes Town",
          "TT-PTF" => "Point Fortin",
          "TT-SFO" => "San Fernando",
          "TT-SGE" => "Sangre Grande",
          "TT-SIP" => "Siparia",
          "TT-SJL" => "San Juan-Laventille",
          "TT-TOB" => "Tobago",
          "TT-TUP" => "Tunapuna-Piarco"
        },
        "TV" => {
          "TV-FUN" => "Funafuti",
          "TV-NMG" => "Nanumaga",
          "TV-NMA" => "Nanumea",
          "TV-NIT" => "Niutao",
          "TV-NUI" => "Nui",
          "TV-NKF" => "Nukufetau",
          "TV-NKL" => "Nukulaelae",
          "TV-VAI" => "Vaitupu"
        },
        "TW" => {
          "TW-CHA" => "Changhua",
          "TW-CYI" => "Chiayi",
          "TW-CYQ" => "Chiayi",
          "TW-HSZ" => "Hsinchu",
          "TW-HSQ" => "Hsinchu",
          "TW-HUA" => "Hualien",
          "TW-KHH" => "Kaohsiung",
          "TW-KEE" => "Keelung",
          "TW-KIN" => "Kinmen",
          "TW-LIE" => "Lienchiang",
          "TW-MIA" => "Miaoli",
          "TW-NAN" => "Nantou",
          "TW-NWT" => "New Taipei",
          "TW-PEN" => "Penghu",
          "TW-PIF" => "Pingtung",
          "TW-TXG" => "Taichung",
          "TW-TNN" => "Tainan",
          "TW-TPE" => "Taipei",
          "TW-TTT" => "Taitung",
          "TW-TAO" => "Taoyuan",
          "TW-ILA" => "Yilan",
          "TW-YUN" => "Yunlin"
        },
        "TZ" => {
          "TZ-01" => "Arusha",
          "TZ-02" => "Dar es Salaam",
          "TZ-03" => "Dodoma",
          "TZ-27" => "Geita",
          "TZ-04" => "Iringa",
          "TZ-05" => "Kagera",
          "TZ-06" => "Kaskazini Pemba",
          "TZ-07" => "Kaskazini Unguja",
          "TZ-28" => "Katavi",
          "TZ-08" => "Kigoma",
          "TZ-09" => "Kilimanjaro",
          "TZ-10" => "Kusini Pemba",
          "TZ-11" => "Kusini Unguja",
          "TZ-12" => "Lindi",
          "TZ-26" => "Manyara",
          "TZ-13" => "Mara",
          "TZ-14" => "Mbeya",
          "TZ-15" => "Mjini Magharibi",
          "TZ-16" => "Morogoro",
          "TZ-17" => "Mtwara",
          "TZ-18" => "Mwanza",
          "TZ-29" => "Njombe",
          "TZ-19" => "Pwani",
          "TZ-20" => "Rukwa",
          "TZ-21" => "Ruvuma",
          "TZ-22" => "Shinyanga",
          "TZ-30" => "Simiyu",
          "TZ-23" => "Singida",
          "TZ-31" => "Songwe",
          "TZ-24" => "Tabora",
          "TZ-25" => "Tanga"
        },
        "UA" => {
          "UA-71" => "Cherkaska oblast",
          "UA-74" => "Chernihivska oblast",
          "UA-77" => "Chernivetska oblast",
          "UA-12" => "Dnipropetrovska oblast",
          "UA-14" => "Donetska oblast",
          "UA-26" => "Ivano-Frankivska oblast",
          "UA-63" => "Kharkivska oblast",
          "UA-65" => "Khersonska oblast",
          "UA-68" => "Khmelnytska oblast",
          "UA-35" => "Kirovohradska oblast",
          "UA-32" => "Kyivska oblast",
          "UA-09" => "Luhanska oblast",
          "UA-46" => "Lvivska oblast",
          "UA-48" => "Mykolaivska oblast",
          "UA-51" => "Odeska oblast",
          "UA-53" => "Poltavska oblast",
          "UA-56" => "Rivnenska oblast",
          "UA-59" => "Sumska oblast",
          "UA-61" => "Ternopilska oblast",
          "UA-05" => "Vinnytska oblast",
          "UA-07" => "Volynska oblast",
          "UA-21" => "Zakarpatska oblast",
          "UA-23" => "Zaporizka oblast",
          "UA-18" => "Zhytomyrska oblast",
          "UA-43" => "Avtonomna Respublika Krym",
          "UA-30" => "Kyiv",
          "UA-40" => "Sevastopol"
        },
        "UG" => {
          "UG-C" => "Central",
          "UG-E" => "Eastern",
          "UG-N" => "Northern",
          "UG-W" => "Western"
        },
        "UM" => {
          "UM-81" => "Baker Island",
          "UM-84" => "Howland Island",
          "UM-86" => "Jarvis Island",
          "UM-67" => "Johnston Atoll",
          "UM-89" => "Kingman Reef",
          "UM-71" => "Midway Islands",
          "UM-76" => "Navassa Island",
          "UM-95" => "Palmyra Atoll",
          "UM-79" => "Wake Island"
        },
        "US" => {
          "US-AL" => "Alabama",
          "US-AK" => "Alaska",
          "US-AZ" => "Arizona",
          "US-AR" => "Arkansas",
          "US-CA" => "California",
          "US-CO" => "Colorado",
          "US-CT" => "Connecticut",
          "US-DE" => "Delaware",
          "US-FL" => "Florida",
          "US-GA" => "Georgia",
          "US-HI" => "Hawaii",
          "US-ID" => "Idaho",
          "US-IL" => "Illinois",
          "US-IN" => "Indiana",
          "US-IA" => "Iowa",
          "US-KS" => "Kansas",
          "US-KY" => "Kentucky",
          "US-LA" => "Louisiana",
          "US-ME" => "Maine",
          "US-MD" => "Maryland",
          "US-MA" => "Massachusetts",
          "US-MI" => "Michigan",
          "US-MN" => "Minnesota",
          "US-MS" => "Mississippi",
          "US-MO" => "Missouri",
          "US-MT" => "Montana",
          "US-NE" => "Nebraska",
          "US-NV" => "Nevada",
          "US-NH" => "New Hampshire",
          "US-NJ" => "New Jersey",
          "US-NM" => "New Mexico",
          "US-NY" => "New York",
          "US-NC" => "North Carolina",
          "US-ND" => "North Dakota",
          "US-OH" => "Ohio",
          "US-OK" => "Oklahoma",
          "US-OR" => "Oregon",
          "US-PA" => "Pennsylvania",
          "US-RI" => "Rhode Island",
          "US-SC" => "South Carolina",
          "US-SD" => "South Dakota",
          "US-TN" => "Tennessee",
          "US-TX" => "Texas",
          "US-UT" => "Utah",
          "US-VT" => "Vermont",
          "US-VA" => "Virginia",
          "US-WA" => "Washington",
          "US-WV" => "West Virginia",
          "US-WI" => "Wisconsin",
          "US-WY" => "Wyoming",
          "US-DC" => "District of Columbia",
          "US-AS" => "American Samoa",
          "US-GU" => "Guam",
          "US-MP" => "Northern Mariana Islands",
          "US-PR" => "Puerto Rico",
          "US-UM" => "United States Minor Outlying Islands",
          "US-VI" => "U.S. Virgin Islands"
        },
        "UY" => {
          "UY-AR" => "Artigas",
          "UY-CA" => "Canelones",
          "UY-CL" => "Cerro Largo",
          "UY-CO" => "Colonia",
          "UY-DU" => "Durazno",
          "UY-FS" => "Flores",
          "UY-FD" => "Florida",
          "UY-LA" => "Lavalleja",
          "UY-MA" => "Maldonado",
          "UY-MO" => "Montevideo",
          "UY-PA" => "Paysandú",
          "UY-RN" => "Río Negro",
          "UY-RV" => "Rivera",
          "UY-RO" => "Rocha",
          "UY-SA" => "Salto",
          "UY-SJ" => "San José",
          "UY-SO" => "Soriano",
          "UY-TA" => "Tacuarembó",
          "UY-TT" => "Treinta y Tres"
        },
        "UZ" => {
          "UZ-TK" => "Toshkent",
          "UZ-AN" => "Andijon",
          "UZ-BU" => "Buxoro",
          "UZ-FA" => "Farg‘ona",
          "UZ-JI" => "Jizzax",
          "UZ-NG" => "Namangan",
          "UZ-NW" => "Navoiy",
          "UZ-QA" => "Qashqadaryo",
          "UZ-SA" => "Samarqand",
          "UZ-SI" => "Sirdaryo",
          "UZ-SU" => "Surxondaryo",
          "UZ-TO" => "Toshkent",
          "UZ-XO" => "Xorazm",
          "UZ-QR" => "Qoraqalpog‘iston Respublikasi"
        },
        "VC" => {
          "VC-01" => "Charlotte",
          "VC-06" => "Grenadines",
          "VC-02" => "Saint Andrew",
          "VC-03" => "Saint David",
          "VC-04" => "Saint George",
          "VC-05" => "Saint Patrick"
        },
        "VE" => {
          "VE-W" => "Dependencias Federales",
          "VE-A" => "Distrito Capital",
          "VE-Z" => "Amazonas",
          "VE-B" => "Anzoátegui",
          "VE-C" => "Apure",
          "VE-D" => "Aragua",
          "VE-E" => "Barinas",
          "VE-F" => "Bolívar",
          "VE-G" => "Carabobo",
          "VE-H" => "Cojedes",
          "VE-Y" => "Delta Amacuro",
          "VE-I" => "Falcón",
          "VE-J" => "Guárico",
          "VE-K" => "Lara",
          "VE-L" => "Mérida",
          "VE-M" => "Miranda",
          "VE-N" => "Monagas",
          "VE-O" => "Nueva Esparta",
          "VE-P" => "Portuguesa",
          "VE-R" => "Sucre",
          "VE-S" => "Táchira",
          "VE-T" => "Trujillo",
          "VE-X" => "Vargas",
          "VE-U" => "Yaracuy",
          "VE-V" => "Zulia"
        },
        "VN" => {
          "VN-44" => "An Giang",
          "VN-43" => "Bà Rịa - Vũng Tàu",
          "VN-54" => "Bắc Giang",
          "VN-53" => "Bắc Kạn",
          "VN-55" => "Bạc Liêu",
          "VN-56" => "Bắc Ninh",
          "VN-50" => "Bến Tre",
          "VN-31" => "Bình Định",
          "VN-57" => "Bình Dương",
          "VN-58" => "Bình Phước",
          "VN-40" => "Bình Thuận",
          "VN-59" => "Cà Mau",
          "VN-04" => "Cao Bằng",
          "VN-33" => "Đắk Lắk",
          "VN-72" => "Đắk Nông",
          "VN-71" => "Điện Biên",
          "VN-39" => "Đồng Nai",
          "VN-45" => "Đồng Tháp",
          "VN-30" => "Gia Lai",
          "VN-03" => "Hà Giang",
          "VN-63" => "Hà Nam",
          "VN-23" => "Hà Tĩnh",
          "VN-61" => "Hải Dương",
          "VN-73" => "Hậu Giang",
          "VN-14" => "Hòa Bình",
          "VN-66" => "Hưng Yên",
          "VN-34" => "Khánh Hòa",
          "VN-47" => "Kiến Giang",
          "VN-28" => "Kon Tum",
          "VN-01" => "Lai Châu",
          "VN-35" => "Lâm Đồng",
          "VN-09" => "Lạng Sơn",
          "VN-02" => "Lào Cai",
          "VN-41" => "Long An",
          "VN-67" => "Nam Định",
          "VN-22" => "Nghệ An",
          "VN-18" => "Ninh Bình",
          "VN-36" => "Ninh Thuận",
          "VN-68" => "Phú Thọ",
          "VN-32" => "Phú Yên",
          "VN-24" => "Quảng Bình",
          "VN-27" => "Quảng Nam",
          "VN-29" => "Quảng Ngãi",
          "VN-13" => "Quảng Ninh",
          "VN-25" => "Quảng Trị",
          "VN-52" => "Sóc Trăng",
          "VN-05" => "Sơn La",
          "VN-37" => "Tây Ninh",
          "VN-20" => "Thái Bình",
          "VN-69" => "Thái Nguyên",
          "VN-21" => "Thanh Hóa",
          "VN-26" => "Thừa Thiên-Huế",
          "VN-46" => "Tiền Giang",
          "VN-51" => "Trà Vinh",
          "VN-07" => "Tuyên Quang",
          "VN-49" => "Vĩnh Long",
          "VN-70" => "Vĩnh Phúc",
          "VN-06" => "Yên Bái",
          "VN-CT" => "Cần Thơ",
          "VN-DN" => "Đà Nẵng",
          "VN-HN" => "Hà Nội",
          "VN-HP" => "Hải Phòng",
          "VN-SG" => "Hồ Chí Minh(local variant is Sai Gon)"
        },
        "VU" => {
          "VU-MAP" => "Malampa",
          "VU-PAM" => "Pénama",
          "VU-SAM" => "Sanma",
          "VU-SEE" => "Shéfa",
          "VU-TAE" => "Taféa",
          "VU-TOB" => "Torba"
        },
        "WF" => {
          "WF-AL" => "Alo",
          "WF-SG" => "Sigave",
          "WF-UV" => "Uvea"
        },
        "WS" => {
          "WS-AA" => "A'ana",
          "WS-AL" => "Aiga-i-le-Tai",
          "WS-AT" => "Atua",
          "WS-FA" => "Fa'asaleleaga",
          "WS-GE" => "Gaga'emauga",
          "WS-GI" => "Gagaifomauga",
          "WS-PA" => "Palauli",
          "WS-SA" => "Satupa'itea",
          "WS-TU" => "Tuamasaga",
          "WS-VF" => "Va'a-o-Fonoti",
          "WS-VS" => "Vaisigano"
        },
        "YE" => {
          "YE-SA" => "Amānat al ‘Āşimah [city]",
          "YE-AB" => "Abyan",
          "YE-AD" => "‘Adan",
          "YE-DA" => "Aḑ Ḑāli‘",
          "YE-BA" => "Al Bayḑā’",
          "YE-HU" => "Al Ḩudaydah",
          "YE-JA" => "Al Jawf",
          "YE-MR" => "Al Mahrah",
          "YE-MW" => "Al Maḩwīt",
          "YE-AM" => "‘Amrān",
          "YE-SU" => "Arkhabīl Suquţrá",
          "YE-DH" => "Dhamār",
          "YE-HD" => "Ḩaḑramawt",
          "YE-HJ" => "Ḩajjah",
          "YE-IB" => "Ibb",
          "YE-LA" => "Laḩij",
          "YE-MA" => "Ma’rib",
          "YE-RA" => "Raymah",
          "YE-SD" => "Şāʻdah",
          "YE-SN" => "Şanʻā’",
          "YE-SH" => "Shabwah",
          "YE-TA" => "Tāʻizz"
        },
        "ZA" => {
          "ZA-EC" => "Eastern Cape",
          "ZA-FS" => "Free State",
          "ZA-GP" => "Gauteng",
          "ZA-KZN" => "Kwazulu-Natal",
          "ZA-LP" => "Limpopo",
          "ZA-MP" => "Mpumalanga",
          "ZA-NC" => "Northern Cape",
          "ZA-NW" => "North-West",
          "ZA-WC" => "Western Cape"
      },
        "ZM" => {
          "ZM-02" => "Central",
          "ZM-08" => "Copperbelt",
          "ZM-03" => "Eastern",
          "ZM-04" => "Luapula",
          "ZM-09" => "Lusaka",
          "ZM-05" => "Northern",
          "ZM-06" => "North-Western",
          "ZM-07" => "Southern",
          "ZM-01" => "Western",
          "ZM-10" => "Muchinga"
        },
        "ZW" => {
          "ZW-BU" => "Bulawayo",
          "ZW-HA" => "Harare",
          "ZW-MA" => "Manicaland",
          "ZW-MC" => "Mashonaland Central",
          "ZW-ME" => "Mashonaland East",
          "ZW-MW" => "Mashonaland West",
          "ZW-MV" => "Masvingo",
          "ZW-MN" => "Matabeleland North",
          "ZW-MS" => "Matabeleland South",
          "ZW-MI" => "Midlands"
        }
      }
    end
  end
end
